#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Void Linux – Installer GUI (GTK3)
Wizard mit klickbaren Schritten:
  1) Willkommen
  2) Sprache (Locale/LANG, glibc-locales)
  3) Tastatur (XKB & Konsole)
  4) Zeitzone
  5) Partitionierung: Überspringen / Manuell (cfdisk) / Btrfs mit Subvolumes (@, @home, @var, @log, @cache, @snapshots)
  6) Benutzer anlegen
  7) Zusammenfassung & Anwenden (mit pkexec, Dry-Run möglich)

Hinweise:
- Änderungen werden erst bei "Anwenden" ausgeführt.
- Btrfs-Modus: FORMATTIERT das angegebene Device, erstellt Subvolumes, mountet nach /mnt und schreibt /mnt/etc/fstab.
- Die restlichen Konfigs (Locale, Keymap, TZ, Benutzer) wirken weiterhin auf /etc des Live-Systems (nicht /mnt).
"""

import gi, os, re, subprocess, shlex
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib, Pango

APP_TITLE = "Void Installer (GUI)"
APP_W, APP_H = 1024, 700
LOG_MAX = 2_000_000

# -------------------------- Utility --------------------------

def list_supported_locales():
    candidates = ["/usr/share/i18n/SUPPORTED", "/etc/locale.gen", "/etc/default/libc-locales"]
    locales = []
    for p in candidates:
        if os.path.isfile(p):
            try:
                with open(p, "r", encoding="utf-8", errors="ignore") as f:
                    for line in f:
                        s = line.strip()
                        if not s or s.startswith("#"):
                            continue
                        loc = s.split()[0]
                        if re.match(r"^[A-Za-z_\.@0-9\-]+$", loc):
                            locales.append(loc)
            except Exception:
                pass
    if not locales:
        locales = ["de_DE.UTF-8","en_US.UTF-8","en_GB.UTF-8","fr_FR.UTF-8","it_IT.UTF-8","es_ES.UTF-8","pl_PL.UTF-8"]
    seen=set(); out=[]
    for l in locales:
        if l not in seen:
            seen.add(l); out.append(l)
    return sorted(out)

def list_xkb_layouts():
    fpath = "/usr/share/X11/xkb/rules/base.lst"
    layouts = []
    if os.path.isfile(fpath):
        try:
            with open(fpath, "r", encoding="utf-8", errors="ignore") as f:
                capture=False
                for line in f:
                    if line.strip().lower()=="! layout":
                        capture=True; continue
                    if capture:
                        if line.startswith("!"): break
                        parts=line.strip().split(None,1)
                        if parts: layouts.append(parts[0])
        except Exception:
            pass
    if not layouts:
        layouts = ["de","us","gb","fr","it","es","pl"]
    return sorted(set(layouts))

def list_timezones(limit=0):
    base = "/usr/share/zoneinfo"
    zones=[]
    for root, dirs, files in os.walk(base):
        if any(skip in root for skip in ["/posix","/right"]):
            continue
        for f in files:
            if f in ("localtime","posixrules","zone.tab","zone1970.tab","leapseconds","tzdata.zi"):
                continue
            rel = os.path.join(root,f).replace(base+"/","")
            zones.append(rel)
            if limit and len(zones)>=limit: break
        if limit and len(zones)>=limit: break
    return sorted(zones)

def run_root_script(script_text, dry_run=False, logger=lambda s: None):
    cmd = ["pkexec","bash","-lc", script_text]
    if dry_run:
        logger("[Dry-Run] pkexec bash -lc <<'EOF'\n"+script_text+"\nEOF\n")
        return 0
    try:
        logger("$ "+" ".join(shlex.quote(c) for c in cmd)+"\n")
        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, universal_newlines=True)
        for line in proc.stdout:
            logger(line)
        return proc.wait()
    except FileNotFoundError:
        logger("Fehler: pkexec nicht gefunden. Bitte polkit installieren.\n"); return 127
    except Exception as e:
        logger(f"Fehler beim Ausführen: {e}\n"); return 1

# -------------------------- Datenmodell --------------------------

class Config:
    def __init__(self):
        self.locale = "de_DE.UTF-8"
        self.xkb_layout = "de"
        self.console_keymap = "de"
        self.timezone = "Europe/Berlin"

        # Partitionierung
        # modes: "skip", "manual", "btrfs"
        self.part_mode = "skip"
        self.part_device = ""          # z.B. /dev/sda2
        self.part_confirm = False      # gefährlicher FORMAT-Guard
        self.part_btrfs_opts = "compress=zstd,noatime,ssd,space_cache=v2"
        self.part_mount_base = "/mnt"
        self.part_subvols = {
            "@": True,
            "@home": True,
            "@var": True,
            "@log": True,
            "@cache": True,
            "@snapshots": False,
        }

        # Benutzer
        self.new_user = {
            "create": True,
            "username": "void",
            "realname": "",
            "password": "",
            "autogroups": True,
        }

        self.dry_run = False

    def summary_lines(self):
        lines = [
            f"Sprache (LANG): {self.locale}",
            f"XKB-Layout: {self.xkb_layout}",
            f"Konsole-Keymap: {self.console_keymap}",
            f"Zeitzone: {self.timezone}",
        ]
        if self.part_mode == "skip":
            lines.append("Partitionierung: übersprungen")
        elif self.part_mode == "manual":
            lines.append(f"Partitionierung: manuell (cfdisk), Ziel: {self.part_device or '—'}")
        else:
            subs = ", ".join([k for k,v in self.part_subvols.items() if v])
            lines += [
                "Partitionierung: Btrfs mit Subvolumes (FORMATIEREN!)",
                f"  Device: {self.part_device or '—'}",
                f"  Mount-Basis: {self.part_mount_base}",
                f"  Mount-Optionen: {self.part_btrfs_opts}",
                f"  Subvolumes: {subs or '—'}",
            ]
        u = self.new_user
        lines.append("Neuer Benutzer: " + ("Ja" if u["create"] else "Nein") + (f" ({u['username']})" if u["create"] else ""))
        lines.append(f"Dry-Run: {'Ja' if self.dry_run else 'Nein'}")
        return lines

# -------------------------- GUI Seiten --------------------------

class PageBase(Gtk.Box):
    def __init__(self, title):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        self.set_border_width(8)
        self.title = title
    def apply_to_config(self, cfg: Config): pass
    def load_from_config(self, cfg: Config): pass

class PageWelcome(PageBase):
    def __init__(self):
        super().__init__("Willkommen")
        lbl = Gtk.Label()
        lbl.set_use_markup(True); lbl.set_xalign(0)
        lbl.set_markup(
            "<span size='large'><b>Willkommen zum Void Installer</b></span>\n"
            "Konfiguriere Sprache, Tastatur, Zeitzone, optional Partitionierung und Benutzer. "
            "Zum Schluss werden alle Änderungen angewendet."
        )
        self.pack_start(lbl, False, False, 0)

class PageLanguage(PageBase):
    def __init__(self):
        super().__init__("Sprache")
        self.combo = Gtk.ComboBoxText()
        locales = list_supported_locales()
        for loc in locales: self.combo.append_text(loc)
        self.combo.set_active(next((i for i,l in enumerate(locales) if l=="de_DE.UTF-8"), 0))
        self.pack_start(Gtk.Label(label="Systemsprache / Locale (LANG)", xalign=0), False, False, 0)
        self.pack_start(self.combo, False, False, 0)
        tip = Gtk.Label(label="Hinweis: Locale wird in /etc/default/libc-locales aktiviert und via "
                               "xbps-reconfigure -f glibc-locales erzeugt.", xalign=0)
        tip.set_line_wrap(True); self.pack_start(tip, False, False, 0)
    def apply_to_config(self, cfg): 
        if self.combo.get_active_text(): cfg.locale = self.combo.get_active_text()

class PageKeyboard(PageBase):
    def __init__(self):
        super().__init__("Tastatur")
        self.combo_xkb = Gtk.ComboBoxText()
        layouts = list_xkb_layouts()
        for l in layouts: self.combo_xkb.append_text(l)
        self.combo_xkb.set_active(next((i for i,l in enumerate(layouts) if l=="de"), 0))
        self.entry_console = Gtk.Entry(); self.entry_console.set_placeholder_text("Konsole-Keymap (z. B. de, de-latin1, us)")
        grid = Gtk.Grid(column_spacing=8, row_spacing=8)
        grid.attach(Gtk.Label(label="XKB-Layout (X11/Wayland):", xalign=0), 0,0,1,1)
        grid.attach(self.combo_xkb, 1,0,1,1)
        grid.attach(Gtk.Label(label="Konsolen-Keymap (/etc/rc.conf → KEYMAP):", xalign=0), 0,1,1,1)
        grid.attach(self.entry_console, 1,1,1,1)
        self.pack_start(grid, False, False, 0)
    def apply_to_config(self, cfg):
        if self.combo_xkb.get_active_text(): cfg.xkb_layout = self.combo_xkb.get_active_text()
        t = self.entry_console.get_text().strip()
        if t: cfg.console_keymap = t

class PageTimezone(PageBase):
    def __init__(self):
        super().__init__("Zeitzone")
        self.entry_filter = Gtk.Entry(); self.entry_filter.set_placeholder_text("Filter (z. B. Europe)")
        self.liststore = Gtk.ListStore(str); self._all_tz = list_timezones()
        for tz in self._all_tz: self.liststore.append([tz])
        self.tree = Gtk.TreeView(model=self.liststore)
        cr = Gtk.CellRendererText(); col = Gtk.TreeViewColumn("Timezone", cr, text=0)
        self.tree.append_column(col); self.tree.set_headers_visible(False)
        self.entry_filter.connect("changed", self.on_filter)
        self.pack_start(Gtk.Label(label="Systemzeitzone (/etc/localtime)", xalign=0), False, False, 0)
        self.pack_start(self.entry_filter, False, False, 0)
        sc = Gtk.ScrolledWindow(); sc.add(self.tree); self.pack_start(sc, True, True, 0)
    def on_filter(self, _):
        q = self.entry_filter.get_text().strip().lower()
        self.liststore.clear()
        for tz in self._all_tz:
            if q in tz.lower(): self.liststore.append([tz])
    def apply_to_config(self, cfg):
        model, it = self.tree.get_selection().get_selected()
        if it: cfg.timezone = model[it][0]

class PagePartition(PageBase):
    def __init__(self):
        super().__init__("Partitionierung")
        # Mode Radios
        self.rb_skip = Gtk.RadioButton.new_with_label_from_widget(None, "Überspringen")
        self.rb_manual = Gtk.RadioButton.new_with_label_from_widget(self.rb_skip, "Manuell (cfdisk)")
        self.rb_btrfs = Gtk.RadioButton.new_with_label_from_widget(self.rb_skip, "Btrfs anlegen (FORMATIEREN!)")
        self.rb_skip.set_active(True)

        # Gemeinsame Felder
        self.entry_device = Gtk.Entry(); self.entry_device.set_placeholder_text("/dev/sda2 oder /dev/nvme0n1p2")
        self.btn_cfdisk = Gtk.Button.new_with_label("cfdisk öffnen")

        # Btrfs-spezifisch
        self.chk_confirm = Gtk.CheckButton(label="Ich weiß, was ich tue (Device wird FORMATIERT)")
        self.entry_opts = Gtk.Entry(); self.entry_opts.set_text("compress=zstd,noatime,ssd,space_cache=v2")
        self.entry_mount = Gtk.Entry(); self.entry_mount.set_text("/mnt")
        # Subvolumes
        self.sub_chks = {}
        for name in ["@","@home","@var","@log","@cache","@snapshots"]:
            chk = Gtk.CheckButton(label=name)
            chk.set_active(name in ["@","@home","@var","@log","@cache"])
            self.sub_chks[name]=chk

        # Layout
        box_modes = Gtk.Box(spacing=10)
        for w in (self.rb_skip,self.rb_manual,self.rb_btrfs): box_modes.pack_start(w, False, False, 0)
        self.pack_start(box_modes, False, False, 0)

        grid = Gtk.Grid(column_spacing=8, row_spacing=8)
        row=0
        grid.attach(Gtk.Label(label="Ziel-Device:", xalign=0), 0,row,1,1); grid.attach(self.entry_device,1,row,1,1); row+=1
        grid.attach(Gtk.Label(label="Manuell:", xalign=0), 0,row,1,1); grid.attach(self.btn_cfdisk,1,row,1,1); row+=1
        grid.attach(Gtk.Label(label="Btrfs Mount-Basis:", xalign=0), 0,row,1,1); grid.attach(self.entry_mount,1,row,1,1); row+=1
        grid.attach(Gtk.Label(label="Btrfs Optionen:", xalign=0), 0,row,1,1); grid.attach(self.entry_opts,1,row,1,1); row+=1
        grid.attach(self.chk_confirm, 1,row,1,1); row+=1

        # Subvol grid
        subgrid = Gtk.Grid(column_spacing=8, row_spacing=4)
        subgrid.attach(Gtk.Label(label="Subvolumes:", xalign=0), 0,0,1,1)
        c=1
        for name, chk in self.sub_chks.items():
            subgrid.attach(chk, c,0,1,1); c+=1
        grid.attach(subgrid, 0,row,2,1); row+=1

        self.pack_start(grid, False, False, 0)

        # Hinweise
        hint = Gtk.Label(
            label="Btrfs-Modus: Device wird formatiert, Subvolumes werden erstellt und unter /mnt gemountet. "
                  "fstab wird in /mnt/etc/fstab geschrieben (Backup fstab.bak).",
            xalign=0
        ); hint.set_line_wrap(True)
        self.pack_start(hint, False, False, 0)

        self.btn_cfdisk.connect("clicked", self.on_cfdisk)

    def on_cfdisk(self, _btn):
        try:
            subprocess.Popen(["xterm","-e","sudo","cfdisk"])
        except FileNotFoundError:
            try:
                subprocess.Popen(["pkexec","cfdisk"])
            except Exception:
                dlg = Gtk.MessageDialog(
                    transient_for=self.get_toplevel(), flags=0,
                    message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE,
                    text="cfdisk konnte nicht gestartet werden."
                ); dlg.format_secondary_text("Installiere ein Terminal (z. B. xterm) oder starte cfdisk separat.")
                dlg.run(); dlg.destroy()

    def apply_to_config(self, cfg: Config):
        if self.rb_btrfs.get_active(): cfg.part_mode = "btrfs"
        elif self.rb_manual.get_active(): cfg.part_mode = "manual"
        else: cfg.part_mode = "skip"
        cfg.part_device = self.entry_device.get_text().strip()
        cfg.part_confirm = self.chk_confirm.get_active()
        cfg.part_btrfs_opts = self.entry_opts.get_text().strip() or "compress=zstd,noatime,ssd,space_cache=v2"
        cfg.part_mount_base = self.entry_mount.get_text().strip() or "/mnt"
        for name, chk in self.sub_chks.items():
            cfg.part_subvols[name] = chk.get_active()

class PageUser(PageBase):
    def __init__(self):
        super().__init__("Benutzer")
        self.switch_create = Gtk.Switch(); self.switch_create.set_active(True)
        self.entry_user = Gtk.Entry(); self.entry_user.set_placeholder_text("Benutzername (z. B. void)")
        self.entry_real = Gtk.Entry(); self.entry_real.set_placeholder_text("Anzeigename (optional)")
        self.entry_pass = Gtk.Entry(); self.entry_pass.set_placeholder_text("Passwort"); self.entry_pass.set_visibility(False)
        self.switch_groups = Gtk.Switch(); self.switch_groups.set_active(True)

        grid = Gtk.Grid(column_spacing=8, row_spacing=8)
        grid.attach(Gtk.Label(label="Benutzer anlegen:", xalign=0), 0,0,1,1); grid.attach(self.switch_create, 1,0,1,1)
        grid.attach(Gtk.Label(label="Benutzername:", xalign=0), 0,1,1,1); grid.attach(self.entry_user, 1,1,1,1)
        grid.attach(Gtk.Label(label="Anzeigename:", xalign=0), 0,2,1,1); grid.attach(self.entry_real, 1,2,1,1)
        grid.attach(Gtk.Label(label="Passwort:", xalign=0), 0,3,1,1); grid.attach(self.entry_pass, 1,3,1,1)
        grid.attach(Gtk.Label(label="Standard-Gruppen (wheel,audio,video,plugdev,lp,scanner…):", xalign=0), 0,4,1,1); grid.attach(self.switch_groups, 1,4,1,1)
        self.pack_start(grid, False, False, 0)

    def apply_to_config(self, cfg):
        u = cfg.new_user
        u["create"] = self.switch_create.get_active()
        u["username"] = self.entry_user.get_text().strip() or "void"
        u["realname"] = self.entry_real.get_text().strip()
        u["password"] = self.entry_pass.get_text()
        u["autogroups"] = self.switch_groups.get_active()

class PageSummary(PageBase):
    def __init__(self, get_cfg_callable, logger_callable):
        super().__init__("Zusammenfassung & Anwenden")
        self.get_cfg = get_cfg_callable
        self.logger = logger_callable

        self.chk_dry = Gtk.Switch(); self.chk_dry.set_active(False)
        drybox = Gtk.Box(spacing=6)
        drybox.pack_start(Gtk.Label(label="Dry-Run (nur anzeigen, nichts ausführen)", xalign=0), False, False, 0)
        drybox.pack_start(self.chk_dry, False, False, 0)
        self.pack_start(drybox, False, False, 0)

        self.textview = Gtk.TextView(); self.textview.set_editable(False)
        self.textview.modify_font(Pango.FontDescription("monospace 10"))
        sc = Gtk.ScrolledWindow(); sc.add(self.textview)
        self.pack_start(sc, True, True, 0)

        self.btn_apply = Gtk.Button.new_with_label("Änderungen anwenden")
        self.btn_apply.connect("clicked", self.on_apply)
        self.pack_start(self.btn_apply, False, False, 0)

    def refresh_summary(self):
        cfg = self.get_cfg()
        buf = self.textview.get_buffer()
        buf.set_text("\n".join("• "+s for s in cfg.summary_lines()))

    def on_apply(self, _btn):
        cfg = self.get_cfg()
        cfg.dry_run = self.chk_dry.get_active()
        script = build_root_script(cfg)
        rc = run_root_script(script, dry_run=cfg.dry_run, logger=self.logger)
        if rc == 0: self.logger("\n[OK] Alle Aktionen erfolgreich.\n")
        else: self.logger(f"\n[FEHLER] Exit {rc}\n")

# -------------------------- Root-Skript Builder --------------------------

def _btrfs_fstab_entries(uuid, opts, subs):
    lines=[]
    # root
    if subs.get("@"):
        lines.append(f"UUID={uuid}\t/\tbtrfs\trw,{opts},subvol=@\t0 0")
    else:
        lines.append(f"UUID={uuid}\t/\tbtrfs\trw,{opts}\t0 0")
    # others
    if subs.get("@home"): lines.append(f"UUID={uuid}\t/home\tbtrfs\trw,{opts},subvol=@home\t0 0")
    if subs.get("@var"): 
        lines.append(f"UUID={uuid}\t/var\tbtrfs\trw,{opts},subvol=@var\t0 0")
        if subs.get("@log"):   lines.append(f"UUID={uuid}\t/var/log\tbtrfs\trw,{opts},subvol=@log\t0 0")
        if subs.get("@cache"): lines.append(f"UUID={uuid}\t/var/cache\tbtrfs\trw,{opts},subvol=@cache\t0 0")
    else:
        if subs.get("@log"):   lines.append(f"UUID={uuid}\t/var/log\tbtrfs\trw,{opts},subvol=@log\t0 0")
        if subs.get("@cache"): lines.append(f"UUID={uuid}\t/var/cache\tbtrfs\trw,{opts},subvol=@cache\t0 0")
    if subs.get("@snapshots"): lines.append(f"UUID={uuid}\t/.snapshots\tbtrfs\trw,{opts},subvol=@snapshots\t0 0")
    return lines

def build_root_script(cfg: Config) -> str:
    lines = [
        'set -e',
        'echo "==> Anwenden startet ..."',
        '',
        '# --- 1) Locale/LANG setzen (live System) ---',
        f'LANG_SET="{shlex.quote(cfg.locale)}"',
        'echo "   - /etc/locale.conf aktualisieren"',
        'printf "LANG=%s\\n" "$LANG_SET" > /etc/locale.conf || true',
        'if [ -f /etc/default/libc-locales ]; then',
        '  sed -i "s/^#\\s*${LANG_SET}\\b/${LANG_SET}/" /etc/default/libc-locales || true',
        '  grep -q "^${LANG_SET}\\b" /etc/default/libc-locales || echo "${LANG_SET}" >> /etc/default/libc-locales',
        '  xbps-reconfigure -f glibc-locales || true',
        'fi',
        '',
        '# --- 2) XKB Tastatur (live System) ---',
        f'XKB_LAYOUT="{shlex.quote(cfg.xkb_layout)}"',
        'mkdir -p /etc/X11/xorg.conf.d',
        'cat > /etc/X11/xorg.conf.d/00-keyboard.conf <<EOF',
        'Section "InputClass"',
        '    Identifier "system-keyboard"',
        '    MatchIsKeyboard "on"',
        '    Option "XkbLayout" "${XKB_LAYOUT}"',
        'EndSection',
        'EOF',
        '',
        '# --- 3) Konsole Keymap (live System) ---',
        f'CONSOLE_KEYMAP="{shlex.quote(cfg.console_keymap)}"',
        'if [ -f /etc/rc.conf ]; then',
        '  if grep -q "^KEYMAP=" /etc/rc.conf; then',
        '    sed -i "s/^KEYMAP=.*/KEYMAP=\\"${CONSOLE_KEYMAP}\\"/" /etc/rc.conf',
        '  else',
        '    echo "KEYMAP=\\"${CONSOLE_KEYMAP}\\"" >> /etc/rc.conf',
        '  fi',
        'else',
        '  printf "KEYMAP=\\"%s\\"\n" "${CONSOLE_KEYMAP}" > /etc/rc.conf',
        'fi',
        '',
        '# --- 4) Zeitzone (live System) ---',
        f'TZ_PATH="{shlex.quote(cfg.timezone)}"',
        'if [ -f "/usr/share/zoneinfo/${TZ_PATH}" ]; then',
        '  ln -sf "/usr/share/zoneinfo/${TZ_PATH}" /etc/localtime',
        '  echo "   - Zeitzone gesetzt: ${TZ_PATH}"',
        'else',
        '  echo "WARN: Zeitzone nicht gefunden: ${TZ_PATH}"',
        'fi',
        '',
        '# --- 5) Partitionierung / Btrfs (Ziel /mnt) ---',
        f'PMODE="{cfg.part_mode}"',
        f'DEV="{shlex.quote(cfg.part_device)}"',
        f'CONFIRM={"1" if cfg.part_confirm else "0"}',
        f'MBASE="{shlex.quote(cfg.part_mount_base)}"',
        f'OPTS="{shlex.quote(cfg.part_btrfs_opts)}"',
        ''
    ]

    # Btrfs Branch
    lines += [
        'if [ "$PMODE" = "btrfs" ]; then',
        '  if [ "$CONFIRM" != "1" ]; then',
        '    echo "FEHLER: Bestätigung fehlt (FORMATIEREN)"; exit 2; fi',
        '  if [ -z "$DEV" ] || [ ! -b "$DEV" ]; then',
        '    echo "FEHLER: Ungültiges Device: $DEV"; exit 2; fi',
        '  echo "==> Btrfs auf $DEV einrichten"',
        '  mkdir -p "$MBASE"',
        '  umount -R "$MBASE" 2>/dev/null || true',
        '  swapoff -a 2>/dev/null || true',
        '  echo "   - mkfs.btrfs -f $DEV"',
        '  mkfs.btrfs -f "$DEV"',
        '  UUID=$(blkid -s UUID -o value "$DEV" || true)',
        '  echo "   - UUID: ${UUID}"',
        '  echo "   - Subvolumes erstellen"',
        '  mount "$DEV" "$MBASE"',
        # subvol creates
        '  ' + ' && '.join([f'(btrfs subvolume create "{cfg.part_mount_base}/{name}")' for name, on in cfg.part_subvols.items() if on]) + ' || true',
        '  umount "$MBASE"',
        '  echo "   - Root (@) mounten"',
        '  mount -o "subvol=@,$OPTS" "$DEV" "$MBASE" || mount "$DEV" "$MBASE"',  # fallback falls @ nicht existiert
        '  mkdir -p "$MBASE/etc" "$MBASE/home" "$MBASE/var" "$MBASE/.snapshots" "$MBASE/var/log" "$MBASE/var/cache"',
        # mount the others if exist/selected
        '  if [ -d "$MBASE/@home" ]; then mkdir -p "$MBASE/home"; fi',
        '  mount -o "subvol=@home,$OPTS" "$DEV" "$MBASE/home" 2>/dev/null || true',
        '  mount -o "subvol=@var,$OPTS" "$DEV" "$MBASE/var" 2>/dev/null || true',
        '  mount -o "subvol=@log,$OPTS" "$DEV" "$MBASE/var/log" 2>/dev/null || true',
        '  mount -o "subvol=@cache,$OPTS" "$DEV" "$MBASE/var/cache" 2>/dev/null || true',
        '  mount -o "subvol=@snapshots,$OPTS" "$DEV" "$MBASE/.snapshots" 2>/dev/null || true',
        '  echo "   - fstab schreiben (in $MBASE/etc/fstab)"',
        '  mkdir -p "$MBASE/etc"',
        '  [ -f "$MBASE/etc/fstab" ] && cp -a "$MBASE/etc/fstab" "$MBASE/etc/fstab.bak" || true',
        '  : > "$MBASE/etc/fstab"',
        '  { echo "# /etc/fstab – generiert vom Installer"; } >> "$MBASE/etc/fstab"',
    ]

    # fstab content generation via shell here-string isn't trivial; we generate with Python now
    # We'll append a marker and then echo each line in shell.
    # Compose entries in Python, insert as echo lines.
    # Build entries now:
    # But we don't have UUID here in Python; shell extracts UUID variable. We'll use shell here-doc with ${UUID}.

    def echo(line): return f'  echo {shlex.quote(line)} >> "$MBASE/etc/fstab"'
    # We will write entries using ${UUID} and $OPTS with subvols
    fstab_shell = [
        echo('UUID=${UUID}\t/\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@\t0 0') if cfg.part_subvols.get("@") else echo('UUID=${UUID}\t/\tbtrfs\trw,'+cfg.part_btrfs_opts+'\t0 0'),
    ]
    if cfg.part_subvols.get("@home"):
        fstab_shell.append(echo('UUID=${UUID}\t/home\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@home\t0 0'))
    if cfg.part_subvols.get("@var"):
        fstab_shell.append(echo('UUID=${UUID}\t/var\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@var\t0 0'))
    if cfg.part_subvols.get("@log"):
        fstab_shell.append(echo('UUID=${UUID}\t/var/log\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@log\t0 0'))
    if cfg.part_subvols.get("@cache"):
        fstab_shell.append(echo('UUID=${UUID}\t/var/cache\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@cache\t0 0'))
    if cfg.part_subvols.get("@snapshots"):
        fstab_shell.append(echo('UUID=${UUID}\t/.snapshots\tbtrfs\trw,'+cfg.part_btrfs_opts+',subvol=@snapshots\t0 0'))

    lines += fstab_shell

    lines += [
        '  echo "   - Btrfs-Setup abgeschlossen."',
        'fi',
        ''
    ]

    # 6) Benutzer anlegen (live System)
    if cfg.new_user["create"]:
        u = cfg.new_user
        lines += [
            '# --- 6) Benutzer (live System) ---',
            f'USER_NAME="{shlex.quote(u["username"])}"',
            f'REAL_NAME="{shlex.quote(u["realname"])}"',
            f'PASSWD_RAW="{shlex.quote(u["password"])}"',
            'GROUPS="wheel,audio,video,cdrom,plugdev,lp,scanner,network"',
            'if [ -n "$USER_NAME" ]; then',
            '  if id "$USER_NAME" >/dev/null 2>&1; then',
            '    echo "   - Benutzer existiert bereits: $USER_NAME"',
            '  else',
            '    useradd -m ${REAL_NAME:+-c "$REAL_NAME"} -G "$GROUPS" "$USER_NAME"',
            '    echo "$USER_NAME:$PASSWD_RAW" | chpasswd',
            '    echo "   - Benutzer angelegt: $USER_NAME"',
            '  fi',
            'fi',
            ''
        ]

    lines += ['echo "==> Anwenden beendet."']
    return "\n".join(lines)

# -------------------------- Hauptfenster --------------------------

class InstallerWindow(Gtk.Window):
    def __init__(self):
        super().__init__(title=APP_TITLE)
        self.set_default_size(APP_W, APP_H); self.set_border_width(8)
        self.cfg = Config()

        root = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6); self.add(root)

        # Header: Seitennavigation
        nav = Gtk.Box(spacing=6); self.btns=[]
        for idx, txt in enumerate(["Willkommen","Sprache","Tastatur","Zeitzone","Partition","Benutzer","Zusammenfassung"]):
            b = Gtk.Button.new_with_label(txt); b.connect("clicked", self.on_nav_click, idx)
            self.btns.append(b); nav.pack_start(b, True, True, 0)
        root.pack_start(nav, False, False, 0)

        self.stack = Gtk.Stack(); self.stack.set_transition_type(Gtk.StackTransitionType.SLIDE_LEFT_RIGHT); self.stack.set_transition_duration(250)

        self.page_welcome = PageWelcome()
        self.page_lang    = PageLanguage()
        self.page_kbd     = PageKeyboard()
        self.page_tz      = PageTimezone()
        self.page_part    = PagePartition()
        self.page_user    = PageUser()
        self.page_summary = PageSummary(self.get_cfg, self.log)

        self.stack.add_titled(self.page_welcome, "welcome", "Willkommen")
        self.stack.add_titled(self.page_lang,    "lang",    "Sprache")
        self.stack.add_titled(self.page_kbd,     "kbd",     "Tastatur")
        self.stack.add_titled(self.page_tz,      "tz",      "Zeitzone")
        self.stack.add_titled(self.page_part,    "part",    "Partition")
        self.stack.add_titled(self.page_user,    "user",    "Benutzer")
        self.stack.add_titled(self.page_summary, "summary", "Zusammenfassung")

        paned = Gtk.Paned(orientation=Gtk.Orientation.VERTICAL); paned.add1(self.stack)

        # Log-Toolbar
        tools = Gtk.Box(spacing=6)
        self.entry_search = Gtk.Entry(); self.entry_search.set_placeholder_text("Suche im Log…")
        btn_find = Gtk.Button.new_with_label("Suchen"); btn_find.connect("clicked", self.on_log_search)
        btn_copy = Gtk.Button.new_with_label("Alles kopieren"); btn_copy.connect("clicked", self.on_log_copy)
        btn_save = Gtk.Button.new_with_label("Log speichern…"); btn_save.connect("clicked", self.on_log_save)
        for w in (self.entry_search, btn_find, btn_copy, btn_save): tools.pack_start(w, False, False, 0)

        # Log-View
        self.textbuf = Gtk.TextBuffer()
        text_log = Gtk.TextView(buffer=self.textbuf); text_log.set_editable(False)
        text_log.modify_font(Pango.FontDescription("monospace 10"))
        sc = Gtk.ScrolledWindow(); sc.add(text_log)

        log_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6); log_box.set_border_width(6)
        log_box.pack_start(Gtk.Label(label="Status/Log", xalign=0), False, False, 0)
        log_box.pack_start(tools, False, False, 0)
        log_box.pack_start(sc, True, True, 0)
        paned.add2(log_box); root.pack_start(paned, True, True, 0)

        self.go(0); self.connect("destroy", Gtk.main_quit)

    # Nav
    def on_nav_click(self, _btn, idx):
        self.collect_current_page(); self.go(idx)
    def go(self, idx):
        pages = ["welcome","lang","kbd","tz","part","user","summary"]
        idx = max(0, min(idx, len(pages)-1)); name = pages[idx]
        if name == "summary":
            self.collect_all_pages(); self.page_summary.refresh_summary()
        self.stack.set_visible_child_name(name)
        for i,b in enumerate(self.btns):
            b.set_sensitive(True)
            (b.get_style_context().add_class if i==idx else b.get_style_context().remove_class)("suggested-action")

    def collect_current_page(self):
        child = self.stack.get_visible_child()
        if hasattr(child, "apply_to_config"): child.apply_to_config(self.cfg)
    def collect_all_pages(self):
        for ch in (self.page_lang, self.page_kbd, self.page_tz, self.page_part, self.page_user):
            ch.apply_to_config(self.cfg)
    def get_cfg(self): return self.cfg

    # Log
    def log(self, text):
        end = self.textbuf.get_end_iter(); self.textbuf.insert(end, text)
        if self.textbuf.get_char_count() > LOG_MAX:
            start = self.textbuf.get_start_iter()
            cut = self.textbuf.get_iter_at_offset(self.textbuf.get_char_count()-LOG_MAX)
            self.textbuf.delete(start, cut)

    # Log-Tools
    def on_log_search(self, _btn):
        q = self.entry_search.get_text(); ifnot = (not q)
        if ifnot: return
        start = self.textbuf.get_start_iter()
        m = start.forward_search(q, 0, None)
        if m:
            a,b = m; self.textbuf.select_range(a,b)
    def on_log_copy(self, _btn):
        start,end = self.textbuf.get_start_iter(), self.textbuf.get_end_iter()
        text = self.textbuf.get_text(start,end,True)
        try:
            from gi.repository import Gdk
            Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD).set_text(text, -1)
        except Exception:
            pass
        self.log("[Kopiert]\n")
    def on_log_save(self, _btn):
        dlg = Gtk.FileChooserDialog(
            title="Log speichern", parent=self,
            action=Gtk.FileChooserAction.SAVE,
            buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL, Gtk.STOCK_SAVE, Gtk.ResponseType.OK),
        )
        dlg.set_current_name("installer.log")
        try:
            if dlg.run() == Gtk.ResponseType.OK:
                path = dlg.get_filename()
                start,end = self.textbuf.get_start_iter(), self.textbuf.get_end_iter()
                text = self.textbuf.get_text(start,end,True)
                try:
                    with open(path,"w",encoding="utf-8") as f: f.write(text)
                    self.log(f"[Gespeichert: {path}]\n")
                except Exception as e:
                    err = Gtk.MessageDialog(transient_for=self, flags=0,
                        message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE,
                        text="Konnte Log nicht speichern")
                    err.format_secondary_text(str(e)); err.run(); err.destroy()
        finally:
            dlg.destroy()

# -------------------------- main --------------------------

if __name__ == "__main__":
    win = InstallerWindow()
    win.show_all()
    Gtk.main()

