#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import gi, os, re, subprocess, shlex, traceback, sys, json
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib, Gdk, Pango

APP_TITLE = "Void Installer"
APP_W, APP_H = 1120, 740
LOG_MAX = 2_000_000

# -------------------------- Styling / Theme --------------------------

APP_CSS = """
* { -GtkWidget-focus-padding: 0; -GtkWidget-focus-line-width: 0; }
window, dialog { background: @theme_base_color; }
.headerbar { padding: 8px 10px; border: none; box-shadow: 0 1px 0 rgba(0,0,0,0.06); }
.headerbar .title { font-weight: 800; letter-spacing: .2px; }
.headerbar .subtitle { opacity: .7; }

.sidebar { background: shade(@theme_base_color, 0.98); border-right: 1px solid alpha(@theme_fg_color, .08); }
.sidebar .stack-list-row { padding: 12px 14px; border-radius: 12px; margin: 2px 6px; }
.sidebar .stack-list-row:selected { background: alpha(@theme_selected_bg_color, .25); }
.sidebar .stack-list-row:hover { background: alpha(@theme_selected_bg_color, .12); }

.card { background: @theme_base_color; border: 1px solid alpha(@theme_fg_color, .08);
        border-radius: 14px; padding: 14px; box-shadow: 0 6px 18px -10px rgba(0,0,0,.25); }
.card-title { font-weight: 700; margin-bottom: 8px; }
.section-label { font-weight: 600; opacity: .8; }
.hint { opacity: .7; }

.big-primary { border-radius: 999px; padding: 10px 18px; }
.action-row { margin-top: 8px; }

.entry { border-radius: 10px; }
.combo { border-radius: 10px; }

.logbox { background: shade(@theme_base_color, 0.97); border-top: 1px solid alpha(@theme_fg_color, .08); }
.log-controls > * { margin-right: 6px; }
"""

def load_css(prefer_dark=True):
    settings = Gtk.Settings.get_default()
    if settings and prefer_dark:
        settings.set_property("gtk-application-prefer-dark-theme", True)
    provider = Gtk.CssProvider()
    provider.load_from_data(APP_CSS.encode("utf-8"))
    screen = Gdk.Screen.get_default()
    Gtk.StyleContext.add_provider_for_screen(screen, provider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)

# -------------------------- Helpers --------------------------

def run(cmd):
    return subprocess.check_output(cmd, text=True)

def list_supported_locales():
    """Füllt eine robuste Locale-Liste: SUPPORTED -> locale -a -> Fallback."""
    locales = []

    # 1) /usr/share/i18n/SUPPORTED
    for p in ("/usr/share/i18n/SUPPORTED", "/etc/locale.gen", "/etc/default/libc-locales"):
        if os.path.isfile(p):
            try:
                with open(p, "r", encoding="utf-8", errors="ignore") as f:
                    for line in f:
                        s = line.strip()
                        if not s or s.startswith("#"): continue
                        loc = s.split()[0]
                        if re.match(r"^[A-Za-z_\.@0-9\-]+$", loc):
                            locales.append(loc)
            except Exception:
                pass

    # 2) locale -a (nur UTF-8 bevorzugen)
    try:
        out = run(["locale", "-a"])
        for line in out.splitlines():
            s = line.strip()
            if s and ("utf" in s.lower() or "UTF-8" in s):
                locales.append(s)
    except Exception:
        pass

    # 3) Fallbacks
    if not locales:
        locales = ["de_DE.UTF-8","en_US.UTF-8","en_GB.UTF-8","fr_FR.UTF-8","it_IT.UTF-8","es_ES.UTF-8","pl_PL.UTF-8"]

    # uniq + sort
    seen=set(); out=[]
    for l in locales:
        if l not in seen:
            seen.add(l); out.append(l)
    return sorted(out)

def list_xkb_layouts():
    fpath = "/usr/share/X11/xkb/rules/base.lst"
    layouts = []
    if os.path.isfile(fpath):
        try:
            with open(fpath, "r", encoding="utf-8", errors="ignore") as f:
                capture=False
                for line in f:
                    if line.strip().lower()=="! layout": capture=True; continue
                    if capture:
                        if line.startswith("!"): break
                        parts=line.strip().split(None,1)
                        if parts: layouts.append(parts[0])
        except Exception: pass
    if not layouts: layouts = ["de","us","gb","fr","it","es","pl"]
    return sorted(set(layouts))

def list_timezones():
    base = "/usr/share/zoneinfo"; zones=[]
    for root, dirs, files in os.walk(base):
        if any(skip in root for skip in ["/posix","/right"]): continue
        for f in files:
            if f in ("localtime","posixrules","zone.tab","zone1970.tab","leapseconds","tzdata.zi"): continue
            zones.append(os.path.join(root,f).replace(base+"/",""))
    return sorted(zones)

def list_block_nodes(include_partitions=True):
    """
    Liefert Liste von (path, label) Einträgen für eine ComboBox.
    Nutzt lsblk -J, zeigt Disks und – optional – Partitionen.
    label z.B.: "/dev/nvme0n1p2 — 120G — PART — btrfs — mounted:/"
    """
    try:
        if not shutil.which("lsblk"):
            return []
    except Exception:
        import shutil
        if not shutil.which("lsblk"):
            return []
    try:
        out = run(["lsblk", "-J", "-o", "NAME,TYPE,SIZE,MODEL,PATH,FSTYPE,MOUNTPOINT"])
        data = json.loads(out)
    except Exception:
        return []

    def fmt(item):
        typ = (item.get("type") or "").upper()
        size = item.get("size","")
        model = item.get("model","") or ""
        path = item.get("path","")
        fstype = item.get("fstype","") or ""
        mnt = item.get("mountpoint","") or ""
        bits = [path, "—", size, "—", typ]
        if fstype: bits += ["—", fstype]
        if mnt:    bits += ["—", f"mounted:{mnt}"]
        if model and item.get("type")=="disk":
            bits += ["—", model]
        return path, " ".join(bits)

    items = []
    for b in data.get("blockdevices", []):
        if b.get("type") == "disk":
            items.append(fmt(b))
            if include_partitions:
                for ch in (b.get("children") or []):
                    if ch.get("type") in ("part","lvm","crypt"):
                        items.append(fmt(ch))
    seen=set(); cleaned=[]
    for p,lbl in items:
        if p not in seen:
            seen.add(p); cleaned.append((p,lbl))
    return cleaned

def run_root_script(script_text, dry_run=False, logger=lambda s: None):
    cmd = ["pkexec","bash","-lc", script_text]
    if dry_run:
        logger("[Dry-Run] pkexec bash -lc <<'EOF'\n"+script_text+"\nEOF\n")
        return 0
    try:
        logger("$ "+" ".join(shlex.quote(c) for c in cmd)+"\n")
        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, universal_newlines=True)
        for line in proc.stdout: logger(line)
        return proc.wait()
    except FileNotFoundError:
        logger("Fehler: pkexec nicht gefunden. Bitte polkit installieren.\n"); return 127
    except Exception as e:
        logger(f"Fehler beim Ausführen: {e}\n"); return 1

# -------------------------- Config --------------------------

class Config:
    def __init__(self):
        self.locale = "de_DE.UTF-8"
        self.xkb_layout = "de"
        self.console_keymap = "de"
        self.timezone = "Europe/Berlin"
        # Partition
        self.part_mode = "skip"   # skip | manual | btrfs
        self.part_device = ""
        self.part_confirm = False
        self.part_btrfs_opts = "compress=zstd,noatime,ssd,space_cache=v2"
        self.part_mount_base = "/mnt"
        self.part_subvols = {"@":True,"@home":True,"@var":True,"@log":True,"@cache":True,"@snapshots":False}
        # Basis-System Installation
        self.install_base = False
        self.base_repo = ""   # leer = host-repos in Ziel kopieren & nutzen
        self.base_packages = "base-system btrfs-progs"
        # User (root bleibt unberührt)
        self.new_user = {"create": True, "username":"void", "realname":"", "password":"", "password2":"", "autogroups":True, "target": True}
        self.dry_run = False
    def summary_lines(self):
        L = [
            f"Sprache (LANG): {self.locale}",
            f"XKB-Layout: {self.xkb_layout}",
            f"Konsole-Keymap: {self.console_keymap}",
            f"Zeitzone: {self.timezone}",
        ]
        if self.part_mode=="skip":
            L.append("Partitionierung: übersprungen")
        elif self.part_mode=="manual":
            L.append(f"Partitionierung: manuell (cfdisk), Ziel: {self.part_device or '—'}")
        else:
            subs=", ".join(k for k,v in self.part_subvols.items() if v)
            L += [
                "Partitionierung: Btrfs (FORMATIEREN!)",
                f"  Device: {self.part_device or '—'}",
                f"  Mount-Basis: {self.part_mount_base}",
                f"  Mount-Optionen: {self.part_btrfs_opts}",
                f"  Subvolumes: {subs or '—'}",
            ]
        L.append("Basis-System: " + ("installieren nach /mnt" if self.install_base else "nicht installieren"))
        if self.install_base:
            L.append(f"  Repo: {self.base_repo or '(Host-Repos ins Ziel kopieren)'}")
            L.append(f"  Pakete: {self.base_packages}")
        u=self.new_user
        L.append("Neuer Benutzer: " + ("Ja" if u["create"] else "Nein") + (f" ({u['username']})" if u["create"] else ""))
        L.append(f"  Ziel: {'/mnt (chroot)'}" if u["target"] else "  Ziel: Live-System")
        L.append(f"Dry-Run: {'Ja' if self.dry_run else 'Nein'}")
        return L

# -------------------------- UI Basisklassen --------------------------

class PageBase(Gtk.Box):
    def __init__(self, title, subtitle=None):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=14)
        self.set_border_width(16)
        # Card wrapper
        card = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        card.get_style_context().add_class("card")
        ttl = Gtk.Label()
        ttl.set_use_markup(True); ttl.set_xalign(0)
        ttl.set_markup(f"<span size='x-large' weight='bold'>{GLib.markup_escape_text(title)}</span>")
        card.pack_start(ttl, False, False, 0)
        if subtitle:
            sub = Gtk.Label(label=subtitle, xalign=0); sub.get_style_context().add_class("hint"); sub.set_line_wrap(True)
            card.pack_start(sub, False, False, 0)
        self._card = card
        self.pack_start(card, True, True, 0)
    def body(self): return self._card
    def apply_to_config(self, cfg: Config): pass
    def load_from_config(self, cfg: Config): pass

# -------------------------- Pages --------------------------

class PageWelcome(PageBase):
    def __init__(self):
        super().__init__("Willkommen", "Richte Sprache, Tastatur, Zeitzone, Partitionen und (optional) das Basis-System ein. Alle Schritte werden am Ende angewendet.")
        lbl = Gtk.Label()
        lbl.set_use_markup(True); lbl.set_xalign(0); lbl.set_line_wrap(True)
        lbl.set_markup("• <b>Links</b> die Schritte wählen\n• Änderungen erst bei <b>„Anwenden“</b>\n• Btrfs-Modus formatiert das Ziel-Device ⚠️")
        self.body().pack_start(lbl, False, False, 0)

class PageLanguage(PageBase):
    def __init__(self):
        super().__init__("Sprache", "Systemsprache / Locale (LANG)")
        box = self.body()

        # Auswahl + manuell + Refresh
        self.combo = Gtk.ComboBoxText(); self.combo.get_style_context().add_class("combo")
        self.entry_custom = Gtk.Entry(); self.entry_custom.set_placeholder_text("oder manuell: z. B. de_DE.UTF-8")
        self.entry_custom.get_style_context().add_class("entry")
        self.btn_refresh = Gtk.Button.new_with_label("Aktualisieren")

        row = Gtk.Box(spacing=8)
        row.pack_start(Gtk.Label(label="Locale auswählen:", xalign=0), False, False, 0)
        row.pack_start(self.combo, False, False, 0)
        row.pack_start(self.btn_refresh, False, False, 0)
        box.pack_start(row, False, False, 0)
        box.pack_start(self._row("Freitext:", self.entry_custom), False, False, 0)

        hint = Gtk.Label(label="Void: /etc/default/libc-locales + xbps-reconfigure -f glibc-locales", xalign=0)
        hint.get_style_context().add_class("hint"); hint.set_line_wrap(True)
        box.pack_start(hint, False, False, 0)

        # initial füllen
        self._fill_locales()
        self.btn_refresh.connect("clicked", lambda *_: self._fill_locales())

    def _row(self, label, widget):
        hb = Gtk.Box(spacing=8)
        hb.pack_start(Gtk.Label(label=label, xalign=0), False, False, 0)
        hb.pack_start(widget, False, False, 0)
        return hb

    def _fill_locales(self):
        self.combo.get_model().clear()
        locs = list_supported_locales()
        if not locs:
            self.combo.append_text("— keine Locales gefunden —")
            self.combo.set_active(0)
            return
        for l in locs:
            self.combo.append_text(l)
        # de_DE bevorzugen
        idx = 0
        for i, l in enumerate(locs):
            if l == "de_DE.UTF-8":
                idx = i; break
        self.combo.set_active(idx)

    def apply_to_config(self, cfg):
        txt = self.entry_custom.get_text().strip()
        if txt:
            cfg.locale = txt
        else:
            v = self.combo.get_active_text()
            if v: cfg.locale = v

class PageKeyboard(PageBase):
    def __init__(self):
        super().__init__("Tastatur", "XKB-Layout (Grafik) & Keymap (Konsole)")
        box = self.body()
        self.combo_xkb = Gtk.ComboBoxText(); self.combo_xkb.get_style_context().add_class("combo")
        layouts = list_xkb_layouts()
        for l in layouts: self.combo_xkb.append_text(l)
        self.combo_xkb.set_active(next((i for i,l in enumerate(layouts) if l=="de"), 0))
        self.entry_console = Gtk.Entry(); self.entry_console.set_placeholder_text("z. B. de, de-latin1, us")
        self.entry_console.get_style_context().add_class("entry")
        box.pack_start(self._row("XKB-Layout:", self.combo_xkb), False, False, 0)
        box.pack_start(self._row("Konsole-Keymap:", self.entry_console), False, False, 0)
    def _row(self, label, widget):
        hb = Gtk.Box(spacing=8); hb.pack_start(Gtk.Label(label=label, xalign=0), False, False, 0); hb.pack_start(widget, False, False, 0); return hb
    def apply_to_config(self, cfg):
        if self.combo_xkb.get_active_text(): cfg.xkb_layout = self.combo_xkb.get_active_text()
        t=self.entry_console.get_text().strip()
        if t: cfg.console_keymap = t

class PageTimezone(PageBase):
    def __init__(self):
        super().__init__("Zeitzone", "/etc/localtime → Zoneinfo")
        box = self.body()
        self.entry_filter = Gtk.Entry(); self.entry_filter.set_placeholder_text("Filter, z. B. Europe")
        self.entry_filter.get_style_context().add_class("entry")
        box.pack_start(self._row("Suche:", self.entry_filter), False, False, 0)
        self.liststore = Gtk.ListStore(str); self._all_tz = list_timezones()
        for tz in self._all_tz: self.liststore.append([tz])
        self.tree = Gtk.TreeView(model=self.liststore); self.tree.set_headers_visible(False)
        cr = Gtk.CellRendererText(); col = Gtk.TreeViewColumn("Timezone", cr, text=0); self.tree.append_column(col)
        sc = Gtk.ScrolledWindow(); sc.set_min_content_height(250); sc.add(self.tree)
        box.pack_start(sc, True, True, 0)
        self.entry_filter.connect("changed", self.on_filter)
    def _row(self, l, w): hb=Gtk.Box(spacing=8); hb.pack_start(Gtk.Label(label=l, xalign=0), False, False, 0); hb.pack_start(w, False, False, 0); return hb
    def on_filter(self, _):
        q=self.entry_filter.get_text().strip().lower(); self.liststore.clear()
        for tz in self._all_tz:
            if q in tz.lower(): self.liststore.append([tz])
    def apply_to_config(self, cfg):
        model, it = self.tree.get_selection().get_selected()
        if it: cfg.timezone = model[it][0]

class PagePartition(PageBase):
    def __init__(self):
        super().__init__("Partitionierung", "Überspringen / Manuell (cfdisk) / Btrfs mit Subvolumes (FORMATIEREN!)")
        box = self.body()
        # Modes
        self.rb_skip = Gtk.RadioButton.new_with_label_from_widget(None, "Überspringen")
        self.rb_manual = Gtk.RadioButton.new_with_label_from_widget(self.rb_skip, "Manuell (cfdisk)")
        self.rb_btrfs = Gtk.RadioButton.new_with_label_from_widget(self.rb_skip, "Btrfs (FORMATIEREN!)")
        self.rb_skip.set_active(True)
        modes = Gtk.Box(spacing=10); [modes.pack_start(w, False, False, 0) for w in (self.rb_skip,self.rb_manual,self.rb_btrfs)]
        box.pack_start(modes, False, False, 0)

        # Ziel-Device: Combo + Refresh + Entry
        self.combo_device = Gtk.ComboBoxText(); self.combo_device.get_style_context().add_class("combo")
        self.btn_refresh = Gtk.Button.new_with_label("Aktualisieren")
        self.entry_device = Gtk.Entry(); self.entry_device.set_placeholder_text("/dev/sda2 oder /dev/nvme0n1p2")
        self.entry_device.get_style_context().add_class("entry")
        row = Gtk.Box(spacing=8)
        row.pack_start(Gtk.Label(label="Ziel-Device:", xalign=0), False, False, 0)
        row.pack_start(self.combo_device, False, False, 0)
        row.pack_start(self.btn_refresh, False, False, 0)
        box.pack_start(row, False, False, 0)
        box.pack_start(self._row("Eingabe (manuell):", self.entry_device), False, False, 0)

        # Manuell
        self.btn_cfdisk = Gtk.Button.new_with_label("cfdisk öffnen"); self.btn_cfdisk.get_style_context().add_class("big-primary")
        box.pack_start(self._row("Manuell:", self.btn_cfdisk), False, False, 0)

        # Btrfs
        self.entry_mount = Gtk.Entry(); self.entry_mount.set_text("/mnt"); self.entry_mount.get_style_context().add_class("entry")
        self.entry_opts = Gtk.Entry(); self.entry_opts.set_text("compress=zstd,noatime,ssd,space_cache=v2"); self.entry_opts.get_style_context().add_class("entry")
        self.chk_confirm = Gtk.CheckButton(label="Ich weiß, was ich tue (Device wird FORMATIERT)")
        box.pack_start(self._row("Btrfs Mount-Basis:", self.entry_mount), False, False, 0)
        box.pack_start(self._row("Btrfs Optionen:", self.entry_opts), False, False, 0)
        box.pack_start(self.chk_confirm, False, False, 0)

        # Subvols row
        subgrid = Gtk.FlowBox(); subgrid.set_selection_mode(Gtk.SelectionMode.NONE)
        self.sub_chks={}
        for name, default in [("@",True),("@home",True),("@var",True),("@log",True),("@cache",True),("@snapshots",False)]:
            chk = Gtk.CheckButton(label=name); chk.set_active(default)
            self.sub_chks[name]=chk; subgrid.add(chk)
        card = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6); card.get_style_context().add_class("card")
        cap = Gtk.Label(label="Subvolumes", xalign=0); cap.get_style_context().add_class("section-label")
        card.pack_start(cap, False, False, 0); card.pack_start(subgrid, False, False, 0)
        box.pack_start(card, False, False, 0)

        hint = Gtk.Label(label="fstab wird in /mnt/etc/fstab geschrieben (Backup fstab.bak).", xalign=0)
        hint.get_style_context().add_class("hint"); hint.set_line_wrap(True)
        box.pack_start(hint, False, False, 0)

        # Fill devices + handlers
        self._fill_devices()
        self.btn_refresh.connect("clicked", lambda *_: self._fill_devices())
        self.combo_device.connect("changed", self._on_combo_changed)
        self.btn_cfdisk.connect("clicked", self.on_cfdisk)

    def _row(self, l, w): hb=Gtk.Box(spacing=8); hb.pack_start(Gtk.Label(label=l, xalign=0), False, False, 0); hb.pack_start(w, False, False, 0); return hb

    def _fill_devices(self):
        self.combo_device.get_model().clear()
        items = list_block_nodes(include_partitions=True)
        if not items:
            self.combo_device.append_text("— keine Geräte gefunden —"); self.combo_device.set_active(0); return
        for path, label in items:
            # id=path, text=label
            self.combo_device.append(path, label)
        self.combo_device.set_active(0)

    def _on_combo_changed(self, combo):
        try:
            cid = combo.get_active_id()
            if cid: self.entry_device.set_text(cid)
        except Exception:
            pass

    def on_cfdisk(self, _btn):
        try: subprocess.Popen(["xterm","-e","sudo","cfdisk"])
        except FileNotFoundError:
            try: subprocess.Popen(["pkexec","cfdisk"])
            except Exception:
                dlg = Gtk.MessageDialog(transient_for=self.get_toplevel(), flags=0,
                    message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE,
                    text="cfdisk konnte nicht gestartet werden.")
                dlg.format_secondary_text("Installiere ein Terminal (z. B. xterm) oder starte cfdisk separat."); dlg.run(); dlg.destroy()

    def apply_to_config(self, cfg: Config):
        if self.rb_btrfs.get_active(): cfg.part_mode="btrfs"
        elif self.rb_manual.get_active(): cfg.part_mode="manual"
        else: cfg.part_mode="skip"
        cfg.part_device = self.entry_device.get_text().strip() or (self.combo_device.get_active_id() or "")
        cfg.part_confirm = self.chk_confirm.get_active()
        cfg.part_btrfs_opts = self.entry_opts.get_text().strip() or "compress=zstd,noatime,ssd,space_cache=v2"
        cfg.part_mount_base = self.entry_mount.get_text().strip() or "/mnt"
        for name, chk in self.sub_chks.items(): cfg.part_subvols[name] = chk.get_active()

class PageBaseInstall(PageBase):
    def __init__(self):
        super().__init__("Basis-System", "Optional: Void-Basis nach /mnt installieren (xbps-install -r /mnt)")
        box = self.body()
        self.switch_enable = Gtk.Switch(); self.switch_enable.set_active(False)
        self.entry_repo = Gtk.Entry(); self.entry_repo.set_placeholder_text("Repo-URL (leer = Host-Repos ins Ziel kopieren und nutzen)")
        self.entry_repo.get_style_context().add_class("entry")
        self.entry_pkgs = Gtk.Entry(); self.entry_pkgs.set_text("base-system btrfs-progs")
        self.entry_pkgs.get_style_context().add_class("entry")
        self.switch_user_target = Gtk.Switch(); self.switch_user_target.set_active(True)

        box.pack_start(self._row("Basis-System installieren:", self.switch_enable), False, False, 0)
        box.pack_start(self._row("Repository:", self.entry_repo), False, False, 0)
        box.pack_start(self._row("Pakete:", self.entry_pkgs), False, False, 0)
        box.pack_start(self._row("Benutzer im Ziel (/mnt) anlegen:", self.switch_user_target), False, False, 0)

        hint = Gtk.Label(label="Hinweis: Kopiert XBPS-Keys/Repo-Configs ins Ziel. Danach Locale & Zeitzone im Ziel setzen und Benutzer per chroot anlegen.", xalign=0)
        hint.get_style_context().add_class("hint"); hint.set_line_wrap(True)
        box.pack_start(hint, False, False, 0)

    def _row(self, l, w): hb=Gtk.Box(spacing=8); hb.pack_start(Gtk.Label(label=l, xalign=0), False, False, 0); hb.pack_start(w, True, True, 0); return hb
    def apply_to_config(self, cfg: Config):
        cfg.install_base = self.switch_enable.get_active()
        cfg.base_repo = self.entry_repo.get_text().strip()
        cfg.base_packages = self.entry_pkgs.get_text().strip() or "base-system btrfs-progs"
        cfg.new_user["target"] = self.switch_user_target.get_active()

class PageUser(PageBase):
    def __init__(self):
        super().__init__("Benutzer", "Root bleibt unverändert. Der Benutzer kommt in die Gruppe „wheel“.")
        box = self.body()
        self.switch_create = Gtk.Switch(); self.switch_create.set_active(True)
        self.entry_user = Gtk.Entry(); self.entry_user.set_placeholder_text("Benutzername (z. B. void)")
        self.entry_real = Gtk.Entry(); self.entry_real.set_placeholder_text("Anzeigename (optional)")
        self.entry_pass1 = Gtk.Entry(); self.entry_pass1.set_placeholder_text("Passwort"); self.entry_pass1.set_visibility(False)
        self.entry_pass2 = Gtk.Entry(); self.entry_pass2.set_placeholder_text("Passwort (Wiederholung)"); self.entry_pass2.set_visibility(False)
        self.switch_groups = Gtk.Switch(); self.switch_groups.set_active(True)

        for e in (self.entry_user,self.entry_real,self.entry_pass1,self.entry_pass2):
            e.get_style_context().add_class("entry")

        box.pack_start(self._row("Benutzer anlegen:", self.switch_create), False, False, 0)
        box.pack_start(self._row("Benutzername:", self.entry_user), False, False, 0)
        box.pack_start(self._row("Anzeigename:", self.entry_real), False, False, 0)
        box.pack_start(self._row("Passwort:", self.entry_pass1), False, False, 0)
        box.pack_start(self._row("Passwort (Wdh.):", self.entry_pass2), False, False, 0)
        box.pack_start(self._row("Standard-Gruppen inkl. wheel:", self.switch_groups), False, False, 0)
    def _row(self, l, w): hb=Gtk.Box(spacing=8); hb.pack_start(Gtk.Label(label=l, xalign=0), False, False, 0); hb.pack_start(w, False, False, 0); return hb
    def apply_to_config(self, cfg):
        u = cfg.new_user
        u["create"] = self.switch_create.get_active()
        u["username"] = self.entry_user.get_text().strip() or "void"
        u["realname"] = self.entry_real.get_text().strip()
        u["password"] = self.entry_pass1.get_text()
        u["password2"] = self.entry_pass2.get_text()
        u["autogroups"] = self.switch_groups.get_active()

class PageSummary(PageBase):
    def __init__(self, get_cfg, logger):
        super().__init__("Zusammenfassung & Anwenden", "Prüfe alles in Ruhe. Mit „Anwenden“ werden die Aktionen ausgeführt.")
        self.get_cfg = get_cfg; self.logger = logger
        box = self.body()

        # Dry-run switch (sichtbar machen)
        h = Gtk.Box(spacing=10)
        self.chk_dry = Gtk.Switch(); self.chk_dry.set_active(False)
        h.pack_start(Gtk.Label(label="Dry-Run (nur anzeigen, nichts ausführen)", xalign=0), False, False, 0)
        h.pack_start(self.chk_dry, False, False, 0)
        box.pack_start(h, False, False, 0)

        # Summary text
        self.textview = Gtk.TextView(); self.textview.set_editable(False)
        self.textview.modify_font(Pango.FontDescription("monospace 10"))
        sc = Gtk.ScrolledWindow(); sc.set_min_content_height(220); sc.add(self.textview)
        box.pack_start(sc, True, True, 0)

        # Apply button
        self.btn_apply = Gtk.Button.new_with_label("Änderungen anwenden")
        self.btn_apply.get_style_context().add_class("big-primary")
        self.btn_apply.connect("clicked", self.on_apply)
        box.pack_start(self.btn_apply, False, False, 0)

    def _alert(self, title, text):
        dlg = Gtk.MessageDialog(transient_for=self.get_toplevel(), flags=0,
            message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE, text=title)
        dlg.format_secondary_text(text); dlg.run(); dlg.destroy()

    def refresh_summary(self):
        cfg = self.get_cfg()
        buf = self.textview.get_buffer()
        buf.set_text("\n".join("• "+s for s in cfg.summary_lines()))

    def on_apply(self, _btn):
        cfg = self.get_cfg()
        # validation user
        u = cfg.new_user
        if u.get("create", False):
            if not u.get("username","").strip():
                self._alert("Benutzername fehlt.", "Bitte einen Benutzernamen angeben."); return
            if not u.get("password"):
                self._alert("Benutzer-Passwort fehlt.", "Bitte ein Benutzer-Passwort eingeben."); return
            if u.get("password") != u.get("password2"):
                self._alert("Benutzer-Passwörter stimmen nicht.", "Bitte beide Passwort-Felder identisch ausfüllen."); return

        cfg.dry_run = self.chk_dry.get_active()
        if cfg.dry_run:
            self._alert("Dry-Run ist aktiv", "Es wird nichts verändert – nur die geplanten Befehle angezeigt.")
        script = build_root_script(cfg)
        rc = run_root_script(script, dry_run=cfg.dry_run, logger=self.logger)
        if rc == 0: self.logger("\n[OK] Alle Aktionen erfolgreich.\n")
        else: self.logger(f"\n[FEHLER] Exit {rc}\n")

# -------------------------- Script Builder --------------------------

def build_root_script(cfg: Config) -> str:
    lines = [
        'set -e',
        'echo "==> Anwenden startet ..."',
        '',
        '# Locale (LIVE)',
        f'LANG_SET="{shlex.quote(cfg.locale)}"',
        'printf "LANG=%s\\n" "$LANG_SET" > /etc/locale.conf || true',
        'if [ -f /etc/default/libc-locales ]; then',
        '  sed -i "s/^#\\s*${LANG_SET}\\b/${LANG_SET}/" /etc/default/libc-locales || true',
        '  grep -q "^${LANG_SET}\\b" /etc/default/libc-locales || echo "${LANG_SET}" >> /etc/default/libc-locales',
        '  xbps-reconfigure -f glibc-locales || true',
        'fi',
        '',
        '# XKB (LIVE)',
        f'XKB_LAYOUT="{shlex.quote(cfg.xkb_layout)}"',
        'mkdir -p /etc/X11/xorg.conf.d',
        'cat > /etc/X11/xorg.conf.d/00-keyboard.conf <<EOF',
        'Section "InputClass"',
        '    Identifier "system-keyboard"',
        '    MatchIsKeyboard "on"',
        '    Option "XkbLayout" "${XKB_LAYOUT}"',
        'EndSection',
        'EOF',
        '',
        '# Console keymap (LIVE)',
        f'CONSOLE_KEYMAP="{shlex.quote(cfg.console_keymap)}"',
        'if [ -f /etc/rc.conf ]; then',
        '  if grep -q "^KEYMAP=" /etc/rc.conf; then',
        '    sed -i "s/^KEYMAP=.*/KEYMAP=\\"${CONSOLE_KEYMAP}\\"/" /etc/rc.conf',
        '  else',
        '    echo "KEYMAP=\\"${CONSOLE_KEYMAP}\\"" >> /etc/rc.conf',
        '  fi',
        'else',
        '  printf "KEYMAP=\\"%s\\"\n" "${CONSOLE_KEYMAP}" > /etc/rc.conf',
        'fi',
        '',
        '# Timezone (LIVE)',
        f'TZ_PATH="{shlex.quote(cfg.timezone)}"',
        'if [ -f "/usr/share/zoneinfo/${TZ_PATH}" ]; then',
        '  ln -sf "/usr/share/zoneinfo/${TZ_PATH}" /etc/localtime',
        '  echo "   - Zeitzone gesetzt (live): ${TZ_PATH}"',
        'else',
        '  echo "WARN: Zeitzone nicht gefunden: ${TZ_PATH}"',
        'fi',
        '',
        '# Partition / Btrfs (TARGET /mnt)',
        f'PMODE="{cfg.part_mode}"',
        f'DEV="{shlex.quote(cfg.part_device)}"',
        f'CONFIRM={"1" if cfg.part_confirm else "0"}',
        f'MBASE="{shlex.quote(cfg.part_mount_base)}"',
        f'OPTS="{shlex.quote(cfg.part_btrfs_opts)}"',
        ''
    ]

    # Btrfs Branch
    lines += [
        'if [ "$PMODE" = "btrfs" ]; then',
        '  [ "$CONFIRM" = "1" ] || { echo "FEHLER: Bestätigung fehlt (FORMATIEREN)"; exit 2; }',
        '  [ -b "$DEV" ] || { echo "FEHLER: Ungültiges Device: $DEV"; exit 2; }',
        '  echo "==> Btrfs auf $DEV einrichten"',
        '  mkdir -p "$MBASE"; umount -R "$MBASE" 2>/dev/null || true; swapoff -a 2>/dev/null || true',
        '  echo "   - mkfs.btrfs -f $DEV"; mkfs.btrfs -f "$DEV"',
        '  UUID=$(blkid -s UUID -o value "$DEV" || true)',
        '  echo "   - UUID: ${UUID}"',
        '  mount "$DEV" "$MBASE"',
    ]
    for name, enabled in cfg.part_subvols.items():
        if enabled:
            lines.append(f'  btrfs subvolume create "$MBASE/{name}" || true')
    lines += [
        '  umount "$MBASE"',
        '  echo "   - Root (@) mounten"',
        '  mount -o "subvol=@,$OPTS" "$DEV" "$MBASE" 2>/dev/null || mount "$DEV" "$MBASE"',
        '  mkdir -p "$MBASE/etc" "$MBASE/home" "$MBASE/var" "$MBASE/.snapshots" "$MBASE/var/log" "$MBASE/var/cache"',
        '  mount -o "subvol=@home,$OPTS" "$DEV" "$MBASE/home" 2>/dev/null || true',
        '  mount -o "subvol=@var,$OPTS" "$DEV" "$MBASE/var" 2>/dev/null || true',
        '  mount -o "subvol=@log,$OPTS" "$DEV" "$MBASE/var/log" 2>/dev/null || true',
        '  mount -o "subvol=@cache,$OPTS" "$DEV" "$MBASE/var/cache" 2>/dev/null || true',
        '  mount -o "subvol=@snapshots,$OPTS" "$DEV" "$MBASE/.snapshots" 2>/dev/null || true',
        '  echo "   - fstab schreiben"',
        '  [ -f "$MBASE/etc/fstab" ] && cp -a "$MBASE/etc/fstab" "$MBASE/etc/fstab.bak" || true',
        '  : > "$MBASE/etc/fstab"',
        '  echo "# /etc/fstab – generiert vom Installer" >> "$MBASE/etc/fstab"',
        '  echo -e "UUID=${UUID}\t/\tbtrfs\trw,$OPTS,subvol=@\t0 0" >> "$MBASE/etc/fstab"',
        '  [ -d "$MBASE/home" ] && echo -e "UUID=${UUID}\t/home\tbtrfs\trw,$OPTS,subvol=@home\t0 0" >> "$MBASE/etc/fstab" || true',
        '  [ -d "$MBASE/var" ] && echo -e "UUID=${UUID}\t/var\tbtrfs\trw,$OPTS,subvol=@var\t0 0" >> "$MBASE/etc/fstab" || true',
        '  [ -d "$MBASE/var/log" ] && echo -e "UUID=${UUID}\t/var/log\tbtrfs\trw,$OPTS,subvol=@log\t0 0" >> "$MBASE/etc/fstab" || true',
        '  [ -d "$MBASE/var/cache" ] && echo -e "UUID=${UUID}\t/var/cache\tbtrfs\trw,$OPTS,subvol=@cache\t0 0" >> "$MBASE/etc/fstab" || true',
        '  [ -d "$MBASE/.snapshots" ] && echo -e "UUID=${UUID}\t/.snapshots\tbtrfs\trw,$OPTS,subvol=@snapshots\t0 0" >> "$MBASE/etc/fstab" || true',
        'fi',
        ''
    ]

    # Basis-System Installation (TARGET)
    lines += [
        f'INSTALL_BASE={"1" if cfg.install_base else "0"}',
        f'REPO={shlex.quote(cfg.base_repo) if cfg.base_repo else ""}',
        f'PKGS="{shlex.quote(cfg.base_packages)}"',
        '',
        'if [ "$INSTALL_BASE" = "1" ]; then',
        '  echo "==> Basis-System nach $MBASE installieren"',
        '  mkdir -p "$MBASE/etc" "$MBASE/var/db/xbps/keys"',
        '  # XBPS Keys & Repos vom Host kopieren (wenn vorhanden)',
        '  cp -a /var/db/xbps/keys/* "$MBASE/var/db/xbps/keys/" 2>/dev/null || true',
        '  [ -d /usr/share/xbps.d ] && mkdir -p "$MBASE/usr/share/xbps.d" && cp -a /usr/share/xbps.d/* "$MBASE/usr/share/xbps.d/" 2>/dev/null || true',
        '  [ -d /etc/xbps.d ] && mkdir -p "$MBASE/etc/xbps.d" && cp -a /etc/xbps.d/* "$MBASE/etc/xbps.d/" 2>/dev/null || true',
        '  if [ -n "$REPO" ]; then',
        '    echo "repository=$REPO" > "$MBASE/etc/xbps.d/00-repository-main.conf"',
        '  fi',
        '  echo "   - xbps-install $PKGS"',
        '  xbps-install -Sy -r "$MBASE" ${REPO:+-R "$REPO"} $PKGS',
        '',
        '  echo "   - Locale & Zeitzone im Ziel setzen"',
        '  echo "LANG=$LANG_SET" > "$MBASE/etc/locale.conf"',
        '  if [ -f "$MBASE/etc/default/libc-locales" ]; then',
        '    sed -i "s/^#\\s*${LANG_SET}\\b/${LANG_SET}/" "$MBASE/etc/default/libc-locales" || true',
        '    grep -q "^${LANG_SET}\\b" "$MBASE/etc/default/libc-locales" || echo "${LANG_SET}" >> "$MBASE/etc/default/libc-locales"',
        '    chroot "$MBASE" xbps-reconfigure -f glibc-locales || true',
        '  fi',
        '  if [ -f "/usr/share/zoneinfo/${TZ_PATH}" ]; then',
        '    ln -sf "/usr/share/zoneinfo/${TZ_PATH}" "$MBASE/etc/localtime"',
        '  fi',
        'fi',
        ''
    ]

    # Benutzer – je nach Ziel: chroot(/mnt) oder live
    if cfg.new_user["create"]:
        u = cfg.new_user
        groups = "wheel,audio,video,cdrom,plugdev,lp,scanner,network" if u.get("autogroups", True) else "wheel"
        target = u.get("target", True)
        if target:
            lines += [
                '# Benutzer (TARGET /mnt, via chroot)',
                f'USER_NAME="{shlex.quote(u["username"])}"',
                f'REAL_NAME="{shlex.quote(u.get("realname",""))}"',
                f'PASSWD_RAW="{shlex.quote(u.get("password",""))}"',
                f'GROUPS="{groups}"',
                'if [ -d "$MBASE" ] && mountpoint -q "$MBASE"; then',
                '  if chroot "$MBASE" id "$USER_NAME" >/dev/null 2>&1; then',
                '    echo "   - Benutzer existiert bereits (target): $USER_NAME"',
                '  else',
                '    chroot "$MBASE" useradd -m ${REAL_NAME:+-c "$REAL_NAME"} -G "$GROUPS" "$USER_NAME"',
                '    chroot "$MBASE" /bin/sh -c "echo \\"$USER_NAME:$PASSWD_RAW\\" | chpasswd"',
                '    echo "   - Benutzer (target) angelegt: $USER_NAME (Gruppen: $GROUPS)"',
                '  fi',
                'else',
                '  echo "WARN: Ziel nicht gemountet – Benutzer (target) übersprungen."',
                'fi',
                ''
            ]
        else:
            lines += [
                '# Benutzer (LIVE)',
                f'USER_NAME="{shlex.quote(u["username"])}"',
                f'REAL_NAME="{shlex.quote(u.get("realname",""))}"',
                f'PASSWD_RAW="{shlex.quote(u.get("password",""))}"',
                f'GROUPS="{groups}"',
                'if id "$USER_NAME" >/dev/null 2://1; then',
                '  echo "   - Benutzer existiert bereits (live): $USER_NAME"',
                'else',
                '  useradd -m ${REAL_NAME:+-c "$REAL_NAME"} -G "$GROUPS" "$USER_NAME"',
                '  echo "$USER_NAME:$PASSWD_RAW" | chpasswd',
                '  echo "   - Benutzer (live) angelegt: $USER_NAME (Gruppen: $GROUPS)"',
                'fi',
                ''
            ]

    lines += ['echo "==> Anwenden beendet."']
    return "\n".join(lines)

# -------------------------- App Window --------------------------

class InstallerWindow(Gtk.ApplicationWindow):
    def __init__(self, app):
        super().__init__(application=app, title=APP_TITLE)
        self.set_default_size(APP_W, APP_H)
        load_css(prefer_dark=True)

        self.cfg = Config()

        # Headerbar
        hb = Gtk.HeaderBar(); hb.set_show_close_button(True)
        hb.get_style_context().add_class("headerbar")
        hb.set_title(APP_TITLE); hb.set_subtitle("Einfach & sicher installieren")
        self.set_titlebar(hb)

        # Root layout: Paned → Sidebar + Content, bottom Log
        outer = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=0); self.add(outer)
        top = Gtk.Paned(orientation=Gtk.Orientation.HORIZONTAL)
        outer.pack_start(top, True, True, 0)

        # Sidebar (StackSidebar)
        self.stack = Gtk.Stack()
        self.stack.set_transition_type(Gtk.StackTransitionType.SLIDE_LEFT_RIGHT)
        self.stack.set_transition_duration(220)

        sidebar = Gtk.StackSidebar(stack=self.stack)
        sidebar.set_size_request(280, -1)
        sidebar.get_style_context().add_class("sidebar")

        # Pages
        self.page_welcome = PageWelcome()
        self.page_lang    = PageLanguage()
        self.page_kbd     = PageKeyboard()
        self.page_tz      = PageTimezone()
        self.page_part    = PagePartition()
        self.page_base    = PageBaseInstall()
        self.page_user    = PageUser()
        self.page_summary = PageSummary(self.get_cfg, self.log)

        self.stack.add_titled(self.page_welcome, "welcome", "Willkommen")
        self.stack.add_titled(self.page_lang,    "lang",    "Sprache")
        self.stack.add_titled(self.page_kbd,     "kbd",     "Tastatur")
        self.stack.add_titled(self.page_tz,      "tz",      "Zeitzone")
        self.stack.add_titled(self.page_part,    "part",    "Partition")
        self.stack.add_titled(self.page_base,    "base",    "Basis-System")
        self.stack.add_titled(self.page_user,    "user",    "Benutzer")
        self.stack.add_titled(self.page_summary, "summary", "Zusammenfassung")

        # Put into Paned
        top.add1(sidebar)
        top.add2(self.stack)
        top.set_position(280)

        # Log area (bottom)
        logwrap = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        logwrap.get_style_context().add_class("logbox")
        logwrap.set_border_width(10)
        outer.pack_start(logwrap, False, False, 0)

        tools = Gtk.Box(spacing=6); tools.get_style_context().add_class("log-controls")
        self.entry_search = Gtk.Entry(); self.entry_search.set_placeholder_text("Suche im Log…")
        btn_find = Gtk.Button.new_from_icon_name("edit-find", Gtk.IconSize.BUTTON); btn_find.set_label("Suchen"); btn_find.set_always_show_image(True)
        btn_copy = Gtk.Button.new_from_icon_name("edit-copy", Gtk.IconSize.BUTTON); btn_copy.set_label("Alles kopieren"); btn_copy.set_always_show_image(True)
        btn_save = Gtk.Button.new_from_icon_name("document-save-as", Gtk.IconSize.BUTTON); btn_save.set_label("Log speichern…"); btn_save.set_always_show_image(True)
        btn_find.connect("clicked", self.on_log_search)
        btn_copy.connect("clicked", self.on_log_copy)
        btn_save.connect("clicked", self.on_log_save)
        for w in (self.entry_search, btn_find, btn_copy, btn_save): tools.pack_start(w, False, False, 0)
        logwrap.pack_start(tools, False, False, 0)

        self.textbuf = Gtk.TextBuffer()
        text_log = Gtk.TextView(buffer=self.textbuf); text_log.set_editable(False)
        text_log.modify_font(Pango.FontDescription("monospace 10"))
        sc = Gtk.ScrolledWindow(); sc.set_min_content_height(140); sc.add(text_log)
        logwrap.pack_start(sc, True, True, 0)

        # Stack signal to refresh summary
        self.stack.connect("notify::visible-child", self._on_stack_change)

    # Data plumbing
    def _on_stack_change(self, *_):
        name = self.stack.get_visible_child_name()
        if name == "summary":
            self.collect_all_pages()
            self.page_summary.refresh_summary()

    def collect_all_pages(self):
        for pg in (self.page_lang, self.page_kbd, self.page_tz, self.page_part, self.page_base, self.page_user):
            pg.apply_to_config(self.cfg)

    def get_cfg(self): return self.cfg

    # Log
    def log(self, text):
        end = self.textbuf.get_end_iter(); self.textbuf.insert(end, text)
        if self.textbuf.get_char_count() > LOG_MAX:
            start = self.textbuf.get_start_iter()
            cut = self.textbuf.get_iter_at_offset(self.textbuf.get_char_count()-LOG_MAX)
            self.textbuf.delete(start, cut)

    # Log tools
    def on_log_search(self, _btn):
        q = self.entry_search.get_text()
        if not q: return
        start = self.textbuf.get_start_iter()
        m = start.forward_search(q, 0, None)
        if m:
            a,b = m; self.textbuf.select_range(a,b)
    def on_log_copy(self, _btn):
        start,end = self.textbuf.get_start_iter(), self.textbuf.get_end_iter()
        text = self.textbuf.get_text(start,end,True)
        try:
            Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD).set_text(text, -1)
        except Exception: pass
        self.log("[Kopiert]\n")
    def on_log_save(self, _btn):
        dlg = Gtk.FileChooserDialog(
            title="Log speichern", parent=self,
            action=Gtk.FileChooserAction.SAVE,
            buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL, Gtk.STOCK_SAVE, Gtk.ResponseType.OK),
        ); dlg.set_current_name("installer.log")
        try:
            if dlg.run() == Gtk.ResponseType.OK:
                path = dlg.get_filename()
                start,end = self.textbuf.get_start_iter(), self.textbuf.get_end_iter()
                text = self.textbuf.get_text(start,end,True)
                with open(path,"w",encoding="utf-8") as f: f.write(text)
                self.log(f"[Gespeichert: {path}]\n")
        except Exception as e:
            err = Gtk.MessageDialog(transient_for=self, flags=0,
                message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE,
                text="Konnte Log nicht speichern")
            err.format_secondary_text(str(e)); err.run(); err.destroy()
        finally:
            dlg.destroy()

# -------------------------- GTK Application --------------------------

class App(Gtk.Application):
    def __init__(self):
        super().__init__(application_id="dev.void.installer")
    def do_activate(self):
        win = InstallerWindow(self); win.show_all()

# -------------------------- Entry --------------------------

def _fatal_dialog_and_log(exc_text: str):
    try: open("installer-error.log","w",encoding="utf-8").write(exc_text)
    except Exception: pass
    try:
        dlg = Gtk.MessageDialog(transient_for=None, flags=0,
            message_type=Gtk.MessageType.ERROR, buttons=Gtk.ButtonsType.CLOSE,
            text="Installer konnte nicht gestartet werden")
        dlg.format_secondary_text(exc_text.splitlines()[-1]); dlg.run(); dlg.destroy()
    except Exception: pass

if __name__ == "__main__":
    try:
        app = App()
        app.run(sys.argv)
    except Exception:
        tb = traceback.format_exc(); print(tb, file=sys.stderr)
        _fatal_dialog_and_log(tb); sys.exit(1)

