import gi
gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")
from gi.repository import Gtk, Adw, GLib
import os
import shutil
import subprocess
import threading
import json
from typing import Optional, Dict, Any, List

TARGET_ROOT = "/mnt/void"

class InstallationView(Gtk.Box):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs, orientation=Gtk.Orientation.VERTICAL, spacing=10)
        # (Rest des __init__ bleibt unverändert)
        for s in (self.set_margin_top, self.set_margin_bottom, self.set_margin_start, self.set_margin_end): s(16)
        title = Gtk.Label.new("Installation"); title.add_css_class("title-2"); title.set_halign(Gtk.Align.START); self.append(title)
        self.steps = [
            {"id": "partitioning", "label": "1. Partitionen erstellen & formatieren", "status": "queued"},
            {"id": "mounting", "label": "2. Dateisysteme einhängen", "status": "queued"},
            {"id": "install_base", "label": "3. Basissystem und Pakete installieren", "status": "queued"},
            {"id": "configure_system", "label": "4. System konfigurieren (fstab, Hostname, etc.)", "status": "queued"},
            {"id": "configure_bootloader", "label": "5. Bootloader (GRUB) installieren", "status": "queued"},
        ]
        self.steps_listbox = Gtk.ListBox(); self.steps_listbox.set_selection_mode(Gtk.SelectionMode.NONE); self.steps_listbox.add_css_class("boxed-list"); self.append(self.steps_listbox)
        self._build_steps_ui()
        log_frame = Gtk.Frame(label="Detailliertes Protokoll"); log_frame.set_margin_top(15); self.append(log_frame)
        sc = Gtk.ScrolledWindow(); sc.set_vexpand(True); sc.set_min_content_height(200); log_frame.set_child(sc)
        self.textview = Gtk.TextView(); self.textview.set_editable(False); self.textbuffer = self.textview.get_buffer(); sc.set_child(self.textview)
        self.plan: Dict[str, Any] = {}; self._worker: Optional[threading.Thread] = None; self.target_root = TARGET_ROOT
        self._binds_active = False

    def _build_steps_ui(self):
        """Initialize the steps UI with initial status"""
        self._update_steps_ui()
    def _update_step_status(self, step_id: str, status: str, error_msg: Optional[str] = None):
        """Update the status of a step in the UI"""
        for step in self.steps:
            if step["id"] == step_id:
                step["status"] = status
                if error_msg:
                    step["error_msg"] = error_msg
                break

        # Update the UI
        self._update_steps_ui()

    def _update_steps_ui(self):
        """Update the steps list UI with current status"""
        # Clear existing UI
        child = self.steps_listbox.get_first_child()
        while child:
            self.steps_listbox.remove(child)
            child = self.steps_listbox.get_first_child()

        # Rebuild UI with current status
        for step in self.steps:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
            box.set_margin_top(8)
            box.set_margin_bottom(8)
            box.set_margin_start(12)
            box.set_margin_end(12)

            # Status icon
            status = step["status"]
            if status == "running":
                icon = "⏳"
                css_class = "warning"
            elif status == "success":
                icon = "✅"
                css_class = "success"
            elif status == "failure":
                icon = "❌"
                css_class = "error"
            else:  # queued
                icon = "⏸️"
                css_class = "dim-label"

            icon_label = Gtk.Label.new(icon)
            icon_label.add_css_class(css_class)
            box.append(icon_label)

            # Step label
            step_label = Gtk.Label.new(step["label"])
            step_label.set_halign(Gtk.Align.START)
            step_label.set_hexpand(True)
            if status == "running":
                step_label.add_css_class("warning")
            elif status == "success":
                step_label.add_css_class("success")
            elif status == "failure":
                step_label.add_css_class("error")
            box.append(step_label)

            # Error message if present
            if status == "failure" and step.get("error_msg"):
                error_label = Gtk.Label.new(step["error_msg"])
                error_label.add_css_class("error")
                error_label.set_wrap(True)
                box.append(error_label)

            row.set_child(box)
            self.steps_listbox.append(row)

    def set_plan(self, plan: Dict[str, Any]): self.plan = plan or {}
    def start_installation(self):
        if self._worker and self._worker.is_alive(): return
        self._worker = threading.Thread(target=self._run_install, daemon=True); self._worker.start()

    def _log(self, msg: str):
        def update_log():
            self.textbuffer.insert(self.textbuffer.get_end_iter(), msg + "\n")
            # Auto-scroll to bottom
            end_iter = self.textbuffer.get_end_iter()
            end_mark = self.textbuffer.create_mark(None, end_iter, False)
            self.textview.scroll_mark_onscreen(end_mark)
            self.textbuffer.delete_mark(end_mark)
        GLib.idle_add(update_log)

    def _run(self, cmd: List[str], check=True, capture=False, chroot=False) -> subprocess.CompletedProcess:
        if chroot: cmd = ["chroot", self.target_root] + cmd
        self._log(f"$ {' '.join(cmd)}")
        try:
            if capture:
                result = subprocess.run(cmd, check=check, text=True, capture_output=True)
                if result.stderr:
                    self._log(f"STDERR: {result.stderr}")
                return result
            else:
                result = subprocess.run(cmd, check=check, text=True, capture_output=True)
                if result.stdout:
                    self._log(f"STDOUT: {result.stdout}")
                if result.stderr:
                    self._log(f"STDERR: {result.stderr}")
                return result
        except subprocess.CalledProcessError as e:
            out = (getattr(e, "stdout", "") or "") + (getattr(e, "stderr", "") or "")
            self._log(f"!! BEFEHL FEHLGESCHLAGEN (Exit Code {e.returncode}): {e}")
            if out.strip():
                self._log(f"!! OUTPUT: {out}")
            if check: raise
            return e

    def _is_mounted(self, path: str) -> bool:
        try:
            with open("/proc/mounts", "r") as f:
                for line in f:
                    parts = line.split()
                    if len(parts) >= 2 and parts[1] == path:
                        return True
                return False
        except Exception as e:
            self._log(f"Warnung: Konnte /proc/mounts nicht lesen: {e}")
            return False

    def _mount_partition_safe(self, device: str, mountpoint: str, partition_name: str):
        """Safely mount a partition with comprehensive error handling and recovery strategies"""
        self._log(f"Mounte {partition_name}-Partition: {device} → {mountpoint}")

        # Try multiple recovery strategies if mounting fails
        recovery_strategies = [
            ("Standard Mount", self._try_standard_mount),
            ("Force Mount with fsck", self._try_mount_with_fsck),
            ("Mount with explicit filesystem", self._try_mount_explicit_fs),
            ("Read-only Mount", self._try_readonly_mount)
        ]

        for strategy_name, strategy_func in recovery_strategies:
            try:
                self._log(f"Versuche {strategy_name} für {device}")
                if strategy_func(device, mountpoint, partition_name):
                    self._log(f"✓ {partition_name}-Partition mit {strategy_name} erfolgreich gemountet")
                    return
            except Exception as e:
                self._log(f"✗ {strategy_name} fehlgeschlagen: {e}")
                continue

        # If all strategies failed
        raise Exception(f"Alle Mount-Strategien für {partition_name}-Partition fehlgeschlagen")

    def _try_standard_mount(self, device: str, mountpoint: str, partition_name: str) -> bool:
        """Try standard mounting procedure"""
        # Wait for device to be ready
        if not self._wait_for_device(device):
            return False

        # Check if device exists
        if not os.path.exists(device):
            self._log(f"Gerät {device} existiert nicht")
            return False

        # Check if already mounted
        if self._is_mounted(mountpoint):
            self._log(f"Warnung: {mountpoint} ist bereits gemountet, versuche unmount...")
            self._run(["umount", mountpoint], check=False)

        # Ensure mountpoint exists
        os.makedirs(mountpoint, exist_ok=True)

        # Check filesystem before mounting
        result = self._run(["file", "-s", device], capture=True, check=False)
        fs_info = result.stdout.strip()
        self._log(f"Dateisystem auf {device}: {fs_info}")

        # Attempt mount
        self._run(["mount", device, mountpoint])

        # Verify mount succeeded
        return self._is_mounted(mountpoint)

    def _try_mount_with_fsck(self, device: str, mountpoint: str, partition_name: str) -> bool:
        """Try mounting after filesystem check"""
        try:
            self._log(f"Führe Dateisystem-Check durch auf {device}")
            # Try to repair filesystem first
            self._run(["fsck", "-y", device], check=False)
            return self._try_standard_mount(device, mountpoint, partition_name)
        except:
            return False

    def _try_mount_explicit_fs(self, device: str, mountpoint: str, partition_name: str) -> bool:
        """Try mounting with explicit filesystem type"""
        try:
            # Get filesystem type
            result = self._run(["blkid", "-s", "TYPE", "-o", "value", device], capture=True, check=False)
            fs_type = result.stdout.strip()
            if fs_type:
                self._log(f"Versuche Mount mit explizitem FS-Typ: {fs_type}")
                os.makedirs(mountpoint, exist_ok=True)
                self._run(["mount", "-t", fs_type, device, mountpoint])
                return self._is_mounted(mountpoint)
        except:
            pass
        return False

    def _try_readonly_mount(self, device: str, mountpoint: str, partition_name: str) -> bool:
        """Try read-only mount as last resort"""
        try:
            self._log(f"Versuche Read-Only Mount für {device}")
            os.makedirs(mountpoint, exist_ok=True)
            self._run(["mount", "-o", "ro", device, mountpoint])
            if self._is_mounted(mountpoint):
                self._log(f"Warnung: {partition_name} nur als Read-Only gemountet!")
                return True
        except:
            pass
        return False

    def _wait_for_device(self, device: str, max_wait: int = 30):
        """Wait for device to become available and ready"""
        import time
        self._log(f"Warte auf Gerät {device}...")

        for attempt in range(max_wait):
            if os.path.exists(device):
                # Device exists, check if it's readable
                try:
                    with open(device, 'rb') as f:
                        f.read(512)  # Try to read first sector
                    self._log(f"Gerät {device} ist bereit nach {attempt + 1} Sekunden")
                    return True
                except (OSError, IOError) as e:
                    if attempt < max_wait - 1:
                        time.sleep(1)
                        continue
                    else:
                        self._log(f"Gerät {device} nicht lesbar nach {max_wait}s: {e}")
                        break
            else:
                if attempt < max_wait - 1:
                    time.sleep(1)
                else:
                    self._log(f"Gerät {device} nicht verfügbar nach {max_wait}s")
                    break

        return False

    def _verify_filesystem(self, device: str, expected_fs: str = None) -> bool:
        """Verify filesystem on device is properly formatted and accessible"""
        try:
            self._log(f"Verifiziere Dateisystem auf {device}...")

            # Check if device is accessible
            result = self._run(["file", "-s", device], capture=True, check=False)
            fs_info = result.stdout.strip().lower()

            # Check for filesystem signatures
            if "filesystem" not in fs_info and "ext" not in fs_info and "btrfs" not in fs_info:
                self._log(f"Warnung: Keine erkennbare Dateisystem-Signatur auf {device}")
                return False

            # Try to get filesystem type with blkid
            try:
                result = self._run(["blkid", "-s", "TYPE", "-o", "value", device], capture=True, check=False)
                fs_type = result.stdout.strip()
                if fs_type:
                    self._log(f"Erkanntes Dateisystem: {fs_type}")
                    if expected_fs and fs_type != expected_fs:
                        self._log(f"Warnung: Erwartet {expected_fs}, gefunden {fs_type}")
                else:
                    self._log(f"Kein Dateisystem-Typ von blkid erkannt")
            except:
                self._log(f"blkid konnte Dateisystem-Typ nicht ermitteln")

            return True

        except Exception as e:
            self._log(f"Dateisystem-Verifikation fehlgeschlagen: {e}")
            return False

    def _run_install(self):
        current_step = ""
        try:
            os.makedirs(self.target_root, exist_ok=True)
            mode = self.plan.get("mode")
            if not mode: raise Exception("Plan-Fehler: 'mode' ist nicht gesetzt.")

            current_step = "partitioning"; GLib.idle_add(self._update_step_status, current_step, "running")
            if mode == "manual": self._format_manual_partitions(self.plan)
            elif mode == "erase": self._apply_auto_partitioning_erase(self.plan)
            GLib.idle_add(self._update_step_status, current_step, "success")

            # ... (Rest der Installation)
            current_step = "mounting"; GLib.idle_add(self._update_step_status, current_step, "running")
            self._mount_filesystems(self.plan)
            GLib.idle_add(self._update_step_status, current_step, "success")

            # Install base system and packages
            current_step = "install_base"; GLib.idle_add(self._update_step_status, current_step, "running")
            self._configure_mirror(self.plan)
            self._xbps_install(self.plan)
            GLib.idle_add(self._update_step_status, current_step, "success")

            # Configure system
            current_step = "configure_system"; GLib.idle_add(self._update_step_status, current_step, "running")
            self._generate_fstab(self.plan)
            self._configure_hostname(self.plan)
            self._configure_locale_kbd(self.plan)
            self._configure_user(self.plan)
            self._enable_services(self.plan)
            self._configure_pipewire(self.plan)
            self._configure_flatpak(self.plan)
            GLib.idle_add(self._update_step_status, current_step, "success")

            # Install bootloader
            current_step = "configure_bootloader"; GLib.idle_add(self._update_step_status, current_step, "running")
            self._install_bootloader(self.plan)
            GLib.idle_add(self._update_step_status, current_step, "success")

            self._log("\n=== INSTALLATION ERFOLGREICH ABGESCHLOSSEN ===")

        except Exception as e:
            self._log(f"\n--- FATALER FEHLER BEI SCHRITT '{current_step}' ---"); self._log(str(e))
            GLib.idle_add(self._update_step_status, current_step, "failure", str(e))
        finally:
            self._leave_chroot_mounts()

    ### --- ÜBERARBEITETE AUFRÄUM-METHODE --- ###
    def _unmount_all_on_device(self, device: str):
        self._log(f"Prüfe und deaktiviere alle Mounts und Volumes auf {device}...")
        try:
            # Schritt 1: Alle normalen Partitionen aushängen und Swap deaktivieren
            out = self._run(["lsblk", "-Jpno", "NAME,TYPE,FSTYPE,MOUNTPOINT", device], capture=True).stdout
            data = json.loads(out)
            partitions = []
            def find_partitions(node):
                if node.get("type") == "part": partitions.append(node)
                for child in node.get("children", []): find_partitions(child)
            for dev_info in data.get("blockdevices", []): find_partitions(dev_info)

            # First pass: unmount all mounted partitions
            for part in reversed(partitions):
                path = part["name"]
                mountpoint = part.get("mountpoint")
                if mountpoint and mountpoint != "":
                    self._log(f"Unmounting {path} from {mountpoint}")
                    self._run(["umount", "-f", "-l", path], check=False)
                if part.get("fstype") == "swap":
                    self._log(f"Disabling swap on {path}")
                    self._run(["swapoff", path], check=False)

            # Second pass: force unmount any remaining mounts
            for part in reversed(partitions):
                path = part["name"]
                if self._is_mounted(path):
                    self._log(f"Force unmounting {path}")
                    self._run(["umount", "-f", "-l", path], check=False)

            # Schritt 2: LVM Volume Groups deaktivieren, die dieses Gerät nutzen
            # `pvs` gibt uns die VG für ein PV (Physical Volume)
            # Schritt 2: LVM deaktivieren (falls vorhanden)
            try:
                # Get partitions using lsblk
                out = self._run(["lsblk", "-Jpno", "NAME,TYPE,FSTYPE,MOUNTPOINT", device], capture=True).stdout
                data = json.loads(out)
                partitions = []
                def find_partitions(node):
                    if node.get("type") == "part": partitions.append(node)
                    for child in node.get("children", []): find_partitions(child)
                for dev_info in data.get("blockdevices", []): find_partitions(dev_info)

                self._log(f"Gefundene Partitionen: {[p['name'] for p in partitions]}")

                # Check each partition for LVM
                for partition in partitions:
                    part_device = partition['name']
                    try:
                        pvs_out = self._run(["pvs", "--noheadings", "-o", "vg_name", part_device], capture=True, check=False)
                        if pvs_out.returncode == 0 and pvs_out.stdout.strip():
                            vg_name = pvs_out.stdout.strip()
                            self._log(f"Deaktiviere LVM Volume Group '{vg_name}' von Partition {part_device}...")
                            self._run(["vgchange", "-an", vg_name], check=False)
                    except Exception as e:
                        self._log(f"LVM-Check für {part_device}: {e}")

                # Also try a general LVM scan to catch any missed VGs
                try:
                    self._run(["vgchange", "-an"], check=False)  # Deactivate all VGs
                    self._log("Alle LVM Volume Groups deaktiviert")
                except Exception as e:
                    self._log(f"Generelle LVM-Deaktivierung: {e}")

            except Exception as e:
                self._log(f"LVM-Deaktivierung fehlgeschlagen: {e}")

            # Schritt 3 (Optional, für RAID): mdadm-Arrays stoppen
            # (Diese Logik ist komplexer und wird hier vorerst ausgelassen)

        except Exception as e:
            self._log(f"Warnung beim Versuch, {device} aufzuräumen: {e}")

    def _apply_auto_partitioning_erase(self, plan: Dict[str, Any]):
        device = plan.get("device")
        layout = plan.get("auto_layout", {})
        if not device or not layout: raise ValueError("Plan für automatische Partitionierung unvollständig.")

        # NEUER, ROBUSTERER ABLAUF
        self._unmount_all_on_device(device)

        self._log(f"!!! DESTRUKTIVE AKTION: LÖSCHE ALLE DATEN AUF {device} !!!")

        # Additional safety: Force unmount any remaining mounts and disable swap
        self._log("Deaktiviere alle Swap-Partitionen...")
        self._run(["swapoff", "-a"], check=False)
        self._log("Hänge alle relevanten Dateisysteme aus...")
        self._run(["umount", "-a", "--types", "ext4,ext3,ext2,xfs,btrfs,vfat,ntfs"], check=False)

        # Wait a moment for the system to release the device
        import time
        time.sleep(1)

        # Try to wipe the device with force flag
        try:
            self._run(["wipefs", "-a", "-f", device])
        except subprocess.CalledProcessError:
            self._log("wipefs failed, trying alternative approach...")
            # Alternative: use dd to zero the first few sectors
            self._run(["dd", "if=/dev/zero", f"of={device}", "bs=512", "count=1024"], check=False)
        self._run(["parted", "-s", device, "mklabel", "gpt" if plan.get("uefi") else "msdos"])

        # (Rest der Methode bleibt gleich)
        assignments = {}; part_num = 1; start_pos = "1MiB"
        if plan.get("uefi"):
            esp_size = layout.get("esp_size_mib", 512); end_pos = f"{esp_size + 1}MiB"
            self._run(["parted", "-s", device, "mkpart", "primary", "fat32", start_pos, end_pos])
            self._run(["parted", "-s", device, "set", str(part_num), "esp", "on"])
            assignments[f"{device}{part_num}"] = {"mountpoint": "/boot/efi", "format": True, "format_fs": "vfat"}; part_num += 1; start_pos = end_pos
        if layout.get("use_swap_partition"):
            swap_size = layout.get("swap_partition_gib", 8)
            # Convert swap size to MiB for consistency
            swap_size_mib = swap_size * 1024
            if plan.get("uefi"):
                esp_size = layout.get("esp_size_mib", 512)
                end_pos = f"{esp_size + 1 + swap_size_mib}MiB"
            else:
                end_pos = f"{1 + swap_size_mib}MiB"
            self._run(["parted", "-s", device, "mkpart", "primary", "linux-swap", start_pos, end_pos])
            assignments[f"{device}{part_num}"] = {"mountpoint": "[SWAP]", "format": True, "format_fs": "swap"}; part_num += 1; start_pos = end_pos
        fs_type = layout.get("filesystem", "btrfs")
        if layout.get("use_separate_home"):
            home_percent = layout.get("home_size_percent", 50); root_end_pos = f"{100 - home_percent}%"
            self._run(["parted", "-s", device, "mkpart", "primary", fs_type, start_pos, root_end_pos])
            assignments[f"{device}{part_num}"] = {"mountpoint": "/", "format": True, "format_fs": fs_type}; part_num += 1
            self._run(["parted", "-s", device, "mkpart", "primary", fs_type, root_end_pos, "100%"])
            assignments[f"{device}{part_num}"] = {"mountpoint": "/home", "format": True, "format_fs": fs_type}
        else:
            self._run(["parted", "-s", device, "mkpart", "primary", fs_type, start_pos, "100%"])
            assignments[f"{device}{part_num}"] = {"mountpoint": "/", "format": True, "format_fs": fs_type}
        self._run(["partprobe", device], check=False)

        # Format all partitions that need formatting
        self._log("Formatiere Partitionen...")
        for dev, assign in assignments.items():
            if assign.get("format"):
                fs_type = assign.get("format_fs", "ext4")
                self._make_fs(dev, fs_type)

        self.plan["manual_partitions"] = assignments


    def _get_uuid(self, device: str) -> str:
        try:
            res = self._run(["blkid", "-s", "UUID", "-o", "value", device], capture=True); return res.stdout.strip()
        except subprocess.CalledProcessError: raise Exception(f"Konnte UUID für Gerät {device} nicht ermitteln.")

    def _make_fs(self, device: str, fs_type: str):
        self._log(f"Formatiere {device} als {fs_type}...")
        if fs_type == "btrfs": self._run(["mkfs.btrfs", "-f", device])
        elif fs_type == "ext4": self._run(["mkfs.ext4", "-F", device])
        elif fs_type == "xfs": self._run(["mkfs.xfs", "-f", device])
        elif fs_type == "vfat": self._run(["mkfs.vfat", "-F32", device])
        elif fs_type == "swap": self._run(["mkswap", device])
        else: raise ValueError(f"Unbekanntes Dateisystem: {fs_type}")

    def _format_manual_partitions(self, plan):
        for dev, assign in plan.get("manual_partitions", {}).items():
            if assign.get("format"): self._make_fs(dev, assign.get("format_fs", "ext4"))

    def _mount_filesystems(self, plan):
        assignments = plan.get("manual_partitions", {});
        if not assignments: raise Exception("Keine Partitionen zum Einhängen gefunden.")
        root_part, home_part, esp_part, swap_part = None, None, None, None
        for dev, assign in assignments.items():
            mp = assign.get("mountpoint")
            if mp == "/": root_part = dev
            elif mp == "/home": home_part = dev
            elif mp == "/boot/efi": esp_part = dev
            elif mp == "[SWAP]": swap_part = dev
        if not root_part: raise Exception("Keine Wurzelpartition (/) zugewiesen.")

        # Mount root partition with enhanced error handling
        self._log(f"=== MOUNTING FILESYSTEMS ===")

        # Wait a moment for all partitions to settle
        import time
        time.sleep(3)

        # Verify and mount root partition
        if not self._verify_filesystem(root_part):
            self._log("Warnung: Root-Partition-Dateisystem nicht vollständig verifiziert")
        self._mount_partition_safe(root_part, self.target_root, "Root")

        # Mount home partition if exists
        if home_part:
            home_mp = os.path.join(self.target_root, "home")
            os.makedirs(home_mp, exist_ok=True)
            if not self._verify_filesystem(home_part):
                self._log("Warnung: Home-Partition-Dateisystem nicht vollständig verifiziert")
            self._mount_partition_safe(home_part, home_mp, "Home")

        # Mount ESP partition if exists and UEFI
        if esp_part and plan.get("uefi"):
            esp_mp = os.path.join(self.target_root, "boot/efi")
            os.makedirs(esp_mp, exist_ok=True)
            if not self._verify_filesystem(esp_part, "vfat"):
                self._log("Warnung: ESP-Partition nicht als vfat verifiziert")
            self._mount_partition_safe(esp_part, esp_mp, "EFI System")
        if swap_part: self._run(["swapon", swap_part])

    def _generate_fstab(self, plan):
        self._log("Generiere /etc/fstab...")
        assignments = plan.get("manual_partitions", {})

        # Ensure /etc directory exists
        etc_dir = os.path.join(self.target_root, "etc")
        os.makedirs(etc_dir, exist_ok=True)

        with open(os.path.join(self.target_root, "etc/fstab"), "w") as f:
            f.write("# /etc/fstab: static file system information.\n# <file system> <mount point> <type> <options> <dump> <pass>\n")
            # Wurzelpartition zuerst
            for dev, assign in assignments.items():
                if assign.get("mountpoint") == "/":
                    uuid = self._get_uuid(dev); fs_type = assign.get("format_fs", "ext4")
                    options = "defaults,noatime,compress=zstd" if fs_type == "btrfs" else "defaults,noatime"
                    f.write(f"UUID={uuid}\t/\t{fs_type}\t{options}\t0 1\n"); break
            # Andere Partitionen
            for dev, assign in assignments.items():
                mp = assign.get("mountpoint")
                if mp in ["/home", "/boot/efi"]:
                    uuid = self._get_uuid(dev); fs_type = "vfat" if mp == "/boot/efi" else assign.get("format_fs", "ext4")
                    options = "defaults,noatime" if fs_type != "vfat" else "defaults"
                    f.write(f"UUID={uuid}\t{mp}\t{fs_type}\t{options}\t0 2\n")
                elif mp == "[SWAP]":
                    uuid = self._get_uuid(dev); f.write(f"UUID={uuid}\tnone\tswap\tsw\t0 0\n")
        self._log("fstab wurde geschrieben.")

    def _configure_mirror(self, plan):
        """Configure package mirrors for xbps"""
        self._log("Konfiguriere Paket-Spiegel...")
        # Use default mirror configuration for now
        pass
    def _setup_xbps_config(self):
        """Setup XBPS configuration for target system"""
        self._log("Setting up XBPS configuration...")

        # Check if target root exists
        if not os.path.exists(self.target_root):
            self._log(f"ERROR: Target root {self.target_root} does not exist")
            raise Exception(f"Target root {self.target_root} does not exist")

        # Create necessary directories
        xbps_cache_dir = os.path.join(self.target_root, "var/cache/xbps")
        xbps_db_dir = os.path.join(self.target_root, "var/db/xbps")
        xbps_keys_dir = os.path.join(self.target_root, "var/db/xbps/keys")
        usr_share_xbps_d = os.path.join(self.target_root, "usr/share/xbps.d")
        etc_xbps_d = os.path.join(self.target_root, "etc/xbps.d")

        for dir_path in [xbps_cache_dir, xbps_db_dir, xbps_keys_dir, usr_share_xbps_d, etc_xbps_d]:
            os.makedirs(dir_path, exist_ok=True)
            self._log(f"Created directory: {dir_path}")

        # Copy repository keys
        host_keys_dir = "/var/db/xbps/keys"
        if os.path.exists(host_keys_dir):
            import shutil
            for key_file in os.listdir(host_keys_dir):
                shutil.copy2(os.path.join(host_keys_dir, key_file),
                           os.path.join(xbps_keys_dir, key_file))
            self._log(f"Copied XBPS keys from {host_keys_dir}")

        # Create basic repository configuration
        repo_conf_path = os.path.join(etc_xbps_d, "00-repository-main.conf")
        with open(repo_conf_path, "w") as f:
            f.write("repository=https://repo-default.voidlinux.org/current\n")
            f.write("repository=https://repo-default.voidlinux.org/current/nonfree\n")
            f.write("repository=https://repo-default.voidlinux.org/current/multilib\n")
            f.write("repository=https://repo-default.voidlinux.org/current/multilib/nonfree\n")
        self._log(f"Created repository configuration: {repo_conf_path}")

        # Test network connectivity
        self._log("Testing network connectivity...")
        try:
            import urllib.request
            urllib.request.urlopen("https://repo-default.voidlinux.org", timeout=10)
            self._log("Network connectivity: OK")
        except Exception as net_err:
            self._log(f"Network connectivity test failed: {net_err}")

    def _validate_packages(self, packages):
        """Validate packages exist in repositories and filter out invalid ones"""
        self._log(f"Validating {len(packages)} packages...")
        valid_packages = []
        invalid_packages = []

        for pkg in packages:

            # For essential packages, skip validation to speed up installation
            essential_packages = {"base-system", "grub", "grub-x86_64-efi", "bash", "curl", "git", "NetworkManager"}
            if pkg in essential_packages:
                valid_packages.append(pkg)
                continue

            # Check if package exists in repositories
            try:
                result = subprocess.run(["xbps-query", "-R", pkg],
                                      capture_output=True, text=True)
                if result.returncode == 0:
                    valid_packages.append(pkg)
                else:
                    self._log(f"Package '{pkg}' not found in repositories")
                    invalid_packages.append(pkg)
            except Exception as e:
                self._log(f"Error checking package '{pkg}': {e}, skipping")
                invalid_packages.append(pkg)

        self._log(f"Validation complete: {len(valid_packages)} valid, {len(invalid_packages)} invalid")
        if invalid_packages:
            self._log(f"Invalid packages skipped: {invalid_packages[:10]}{'...' if len(invalid_packages) > 10 else ''}")

        return valid_packages

    def _xbps_install(self, plan):
        pkgs = ["base-system", "grub"] + plan.get("software", [])
        if plan.get("uefi"): pkgs.append("grub-x86_64-efi")
        all_pkgs = sorted(list(set(pkgs)))
        self._log(f"Requested packages ({len(all_pkgs)} total): {all_pkgs}")

        # Setup XBPS configuration first
        try:
            self._setup_xbps_config()
        except Exception as setup_err:
            self._log(f"XBPS setup failed: {setup_err}")
            raise

        # Sync repositories first (without -r flag to sync host)
        self._log("Synchronizing host repositories...")
        try:
            result = subprocess.run(["xbps-install", "-S"],
                                  text=True, capture_output=True)
            if result.returncode != 0:
                self._log(f"Host sync failed: {result.stderr}")
            else:
                self._log("Host repository sync: OK")
        except Exception as host_sync_err:
            self._log(f"Host sync error: {host_sync_err}")

        # Now sync target repositories
        self._log("Synchronizing target repositories...")
        try:
            result = subprocess.run(["xbps-install", "-S", "-r", self.target_root],
                                  text=True, capture_output=True)
            if result.stdout:
                self._log(f"Target sync STDOUT: {result.stdout}")
            if result.stderr:
                self._log(f"Target sync STDERR: {result.stderr}")
            if result.returncode != 0:
                self._log(f"Target sync failed with return code: {result.returncode}")
                raise subprocess.CalledProcessError(result.returncode, result.args)
            self._log("Target repository sync: OK")

        except subprocess.CalledProcessError as e:
            self._log(f"Repository sync failed with exit code {e.returncode}")
            raise

        # Validate packages before installation
        final_pkgs = self._validate_packages(all_pkgs)

        if not final_pkgs:
            self._log("No valid packages to install!")
            raise Exception("No valid packages found after validation")

        # If too many packages failed validation, fall back to minimal set
        if len(final_pkgs) < len(all_pkgs) * 0.7:  # Less than 70% success rate
            self._log(f"Many packages failed validation ({len(all_pkgs) - len(final_pkgs)} failed)")
            self._log("Falling back to minimal essential package set")
            minimal_pkgs = ["base-system", "grub"]
            if plan.get("uefi"):
                minimal_pkgs.append("grub-x86_64-efi")
            # Add some essential packages that are likely to exist
            minimal_pkgs.extend(["NetworkManager", "bash-completion", "curl", "git"])
            final_pkgs = minimal_pkgs
            self._log(f"Using minimal package set: {final_pkgs}")

        # Install packages
        self._log(f"Installing {len(final_pkgs)} validated packages...")
        try:
            # Use Popen for real-time output instead of run with capture_output
            process = subprocess.Popen(["xbps-install", "-uy", "-r", self.target_root] + final_pkgs,
                                     stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                                     text=True, universal_newlines=True)

            # Read output line by line in real-time
            while True:
                output = process.stdout.readline()
                if output == '' and process.poll() is not None:
                    break
                if output:
                    self._log(output.strip())

            # Wait for process to complete and get return code
            return_code = process.poll()
            if return_code != 0:
                self._log(f"Package installation failed with exit code {return_code}")
                raise subprocess.CalledProcessError(return_code, process.args)
            self._log("Package installation: OK")

        except subprocess.CalledProcessError as e:
            self._log(f"Package installation failed with exit code {e.returncode}")
            raise

    def _configure_hostname(self, plan):
        hostname = plan.get("user", {}).get("hostname")
        if hostname:
            with open(os.path.join(self.target_root, "etc/hostname"), "w") as f: f.write(hostname + "\n")

    def _configure_locale_kbd(self, plan):
        """Configure locale and keyboard layout"""
        self._log("Konfiguriere Sprache und Tastatur...")

        # Set locale
        language = plan.get("language", "de_DE.UTF-8")
        try:
            with open(os.path.join(self.target_root, "etc/locale.conf"), "w") as f:
                f.write(f"LANG={language}\n")
            self._log(f"Sprache auf {language} gesetzt")
        except Exception as e:
            self._log(f"Warnung: Konnte Sprache nicht setzen: {e}")

        # Set keyboard layout
        keyboard = plan.get("keyboard")
        if keyboard:
            try:
                with open(os.path.join(self.target_root, "etc/vconsole.conf"), "w") as f:
                    f.write(f"KEYMAP={keyboard}\n")
                self._log(f"Tastaturlayout auf {keyboard} gesetzt")
            except Exception as e:
                self._log(f"Warnung: Konnte Tastaturlayout nicht setzen: {e}")
    def _configure_user(self, plan):
        """Configure user account"""
        user_config = plan.get("user", {})
        if not user_config:
            self._log("Keine Benutzerkonfiguration gefunden, überspringe...")
            return

        username = user_config.get("name")
        password = user_config.get("password")
        groups = user_config.get("groups", ["wheel"])

        if username:
            self._log(f"Erstelle Benutzer: {username}")
            self._enter_chroot_mounts()
            try:
                # Create user
                self._run(["useradd", "-m", "-G", ",".join(groups), username], chroot=True)

                # Set password if provided
                if password:
                    process = subprocess.Popen(
                        ["chroot", self.target_root, "passwd", username],
                        stdin=subprocess.PIPE,
                        text=True
                    )
                    process.communicate(input=f"{password}\n{password}\n")
                    if process.returncode == 0:
                        self._log("Benutzerpasswort gesetzt")
                    else:
                        self._log("Warnung: Konnte Passwort nicht setzen")

                self._log(f"Benutzer {username} erfolgreich erstellt")
            except Exception as e:
                self._log(f"Fehler beim Erstellen des Benutzers: {e}")
            finally:
                self._leave_chroot_mounts()
    def _enable_services(self, plan):
        """Enable system services"""
        self._log("Aktiviere System-Services...")

        # Basic services that should always be enabled
        basic_services = ["dhcpcd", "chronyd"]

        self._enter_chroot_mounts()
        try:
            for service in basic_services:
                try:
                    # Check if service exists
                    service_path = f"/etc/sv/{service}"
                    if os.path.exists(os.path.join(self.target_root, service_path.lstrip('/'))):
                        self._run(["ln", "-sf", service_path, f"/var/service/{service}"], chroot=True)
                        self._log(f"Service {service} aktiviert")
                    else:
                        self._log(f"Service {service} nicht gefunden, überspringe...")
                except Exception as e:
                    self._log(f"Warnung: Konnte Service {service} nicht aktivieren: {e}")
        finally:
            self._leave_chroot_mounts()

    def _configure_pipewire(self, plan):
        if "pipewire" not in plan.get("software", []): return
        self._log("Konfiguriere PipeWire als Standard-Audioserver...")
        self._enter_chroot_mounts()
        try:
            os.makedirs(os.path.join(self.target_root, "etc/alsa/conf.d"), exist_ok=True)
            self._run(["ln", "-sf", "/usr/share/alsa/alsa.conf.d/50-pipewire.conf", "/etc/alsa/conf.d/"], chroot=True)
            self._run(["ln", "-sf", "/usr/share/alsa/alsa.conf.d/99-pipewire-default.conf", "/etc/alsa/conf.d/"], chroot=True)
        finally: self._leave_chroot_mounts()

    def _configure_flatpak(self, plan):
        if "flatpak" not in plan.get("software", []): return
        self._log("Füge Flathub-Repository zu Flatpak hinzu...")
        self._enter_chroot_mounts()
        try:
            self._run(["flatpak", "remote-add", "--if-not-exists", "flathub", "https://dl.flathub.org/repo/flathub.flatpakrepo"], chroot=True)
        finally: self._leave_chroot_mounts()

    def _install_bootloader(self, plan):
        self._log("Installiere GRUB Bootloader..."); self._enter_chroot_mounts()
        try:
            if plan.get("uefi"): self._run(["grub-install", "--target=x86_64-efi", "--efi-directory=/boot/efi", "--bootloader-id=Void"], chroot=True)
            else: self._run(["grub-install", plan["device"]], chroot=True)
            self._run(["grub-mkconfig", "-o", "/boot/grub/grub.cfg"], chroot=True)
        finally: self._leave_chroot_mounts()

    def _enter_chroot_mounts(self):
        # Mount EFI variables first for UEFI systems
        if os.path.exists("/sys/firmware/efi"):
            self._run(["modprobe", "efivarfs"], check=False)
            self._run(["mount", "-t", "efivarfs", "efivarfs", "/sys/firmware/efi/efivars"], check=False)

        for p in ["/dev", "/proc", "/sys"]:
            target_path = os.path.join(self.target_root, p.lstrip('/')); os.makedirs(target_path, exist_ok=True)
            self._run(["mount", "--bind", p, target_path])

        # Mount EFI variables in chroot for UEFI systems
        if os.path.exists("/sys/firmware/efi"):
            efivars_target = os.path.join(self.target_root, "sys/firmware/efi/efivars")
            os.makedirs(efivars_target, exist_ok=True)
            self._run(["mount", "--bind", "/sys/firmware/efi/efivars", efivars_target], check=False)

        # Copy DNS configuration for network connectivity in chroot
        try:
            resolv_source = "/etc/resolv.conf"
            resolv_target = os.path.join(self.target_root, "etc/resolv.conf")
            if os.path.exists(resolv_source):
                os.makedirs(os.path.dirname(resolv_target), exist_ok=True)
                import shutil
                shutil.copy2(resolv_source, resolv_target)
                self._log("DNS configuration copied to chroot")
        except Exception as e:
            self._log(f"Warning: Could not copy DNS config: {e}")

    def _leave_chroot_mounts(self):
        # Remove DNS configuration copy
        try:
            resolv_target = os.path.join(self.target_root, "etc/resolv.conf")
            if os.path.exists(resolv_target):
                os.remove(resolv_target)
        except Exception as e:
            self._log(f"Warning: Could not remove DNS config copy: {e}")

        # Unmount EFI variables from chroot first
        if os.path.exists("/sys/firmware/efi"):
            efivars_target = os.path.join(self.target_root, "sys/firmware/efi/efivars")
            if os.path.exists(efivars_target) and self._is_mounted(efivars_target):
                try: self._run(["umount", efivars_target])
                except subprocess.CalledProcessError: pass

        for p in ["/dev", "/proc", "/sys"]:
            target_path = os.path.join(self.target_root, p.lstrip('/'))
            if os.path.exists(target_path) and self._is_mounted(target_path):
                try: self._run(["umount", "-R", target_path])
                except subprocess.CalledProcessError: pass
