import os
import shlex
import subprocess
from typing import Dict, Any, List, Optional

class InstallerError(Exception):
    pass

def run(cmd: str, check: bool = True, capture: bool = False, env: Optional[dict] = None) -> subprocess.CompletedProcess:
    args = shlex.split(cmd)
    return subprocess.run(args, check=check, text=True, capture_output=capture, env=env)

def blkid_uuid(dev: str) -> Optional[str]:
    try:
        r = run(f"blkid -s UUID -o value {shlex.quote(dev)}", check=True, capture=True)
        uuid = r.stdout.strip()
        return uuid or None
    except subprocess.CalledProcessError:
        return None

def partprobe(dev: str):
    try:
        run(f"partprobe {shlex.quote(dev)}", check=False)
    except Exception:
        pass

def ensure_dir(path: str):
    os.makedirs(path, exist_ok=True)

def write_file(path: str, content: str, mode: int = 0o644):
    ensure_dir(os.path.dirname(path))
    with open(path, "w") as f:
        f.write(content)
    os.chmod(path, mode)

def make_fs_root(root_part: str, fstype: str):
    if fstype == "btrfs":
        run(f"mkfs.btrfs -f {shlex.quote(root_part)}")
    elif fstype == "ext4":
        run(f"mkfs.ext4 -F {shlex.quote(root_part)}")
    elif fstype == "xfs":
        run(f"mkfs.xfs -f {shlex.quote(root_part)}")
    else:
        raise InstallerError(f"Unbekanntes Dateisystem: {fstype}")

def make_esp(esp_part: str):
    run(f"mkfs.vfat -F32 {shlex.quote(esp_part)}")

def mount(device: str, target: str, opts: Optional[str] = None, fstype: Optional[str] = None):
    ensure_dir(target)
    cmd = ["mount"]
    if fstype:
        cmd += ["-t", fstype]
    if opts:
        cmd += ["-o", opts]
    cmd += [device, target]
    run(" ".join(map(shlex.quote, cmd)))

def umount(target: str, lazy: bool = False):
    flag = " -l" if lazy else ""
    run(f"umount{flag} {shlex.quote(target)}", check=False)

def create_btrfs_subvols(mount_point: str, subvols: List[Dict[str, str]]):
    for sv in subvols:
        name = sv["name"]
        run(f"btrfs subvolume create {shlex.quote(os.path.join(mount_point, name))}")

def mount_btrfs_layout(root_part: str, plan: Dict[str, Any], target_root: str):
    """Mount top-level, create selected subvols, then remount with subvols."""
    opts_root = plan.get("mount_options", "") or "noatime,compress=zstd,ssd,space_cache=v2,commit=120"
    # 1) Top-Level mounten
    mount(root_part, target_root, opts=opts_root, fstype="btrfs")
    # 2) Subvolumes anlegen (inkl. @ sicherstellen)
    subvols = plan.get("btrfs", {}).get("subvolumes", [])
    names = [s["name"] for s in subvols]
    if "@" not in names:
        subvols = [{"name": "@", "mountpoint": "/"}] + subvols
    create_btrfs_subvols(target_root, subvols)
    # 3) Top-Level unmounten
    umount(target_root, lazy=False)
    # 4) @ als / mounten
    mount(f"{root_part}", target_root, opts=f"subvol=@,{opts_root}", fstype="btrfs")
    # 5) Rest mounten
    for sv in subvols:
        if sv["name"] == "@":
            continue
        mnt = sv["mountpoint"].lstrip("/")
        ensure_dir(os.path.join(target_root, mnt))
        mount(root_part, os.path.join(target_root, mnt), opts=f"subvol={sv['name']},{opts_root}", fstype="btrfs")

def generate_fstab(plan: Dict[str, Any], root_part: str, esp_part: Optional[str], target_root: str):
    lines = []
    root_uuid = blkid_uuid(root_part)
    fstype = plan["filesystem"]

    if fstype == "btrfs":
        opts = plan.get("mount_options", "noatime,compress=zstd,ssd,space_cache=v2,commit=120")
        lines.append(f"UUID={root_uuid}\t/\tbtrfs\tsubvol=@,{opts}\t0 0")
        for sv in plan.get("btrfs", {}).get("subvolumes", []):
            if sv["name"] == "@":
                continue
            lines.append(f"UUID={root_uuid}\t{sv['mountpoint']}\tbtrfs\tsubvol={sv['name']},{opts}\t0 0")
    else:
        lines.append(f"UUID={root_uuid}\t/\t{fstype}\tnoatime\t0 1")

    if esp_part:
        esp_uuid = blkid_uuid(esp_part)
        ensure_dir(os.path.join(target_root, "boot/efi"))
        lines.append(f"UUID={esp_uuid}\t/boot/efi\tvfat\tumask=0077\t0 2")

    fstab_path = os.path.join(target_root, "etc/fstab")
    write_file(fstab_path, "\n".join(lines) + "\n")

def setup_swap(plan: Dict[str, Any], target_root: str):
    swap = plan.get("swap", {}) or {}
    mode = swap.get("mode", "none")
    if mode == "none":
        return
    if mode == "swappart":
        part = swap.get("partition")
        if not part:
            raise InstallerError("Swap-Partition gewählt, aber keine Partition angegeben.")
        uuid = blkid_uuid(part)
        run(f"mkswap {shlex.quote(part)}")
        with open(os.path.join(target_root, "etc/fstab"), "a") as f:
            f.write(f"UUID={uuid}\tswap\tswap\tdefaults\t0 0\n")
        return
    if mode == "swapfile":
        size_gib = int(swap.get("size_gib", 4))
        swapfile = os.path.join(target_root, "swapfile")
        run(f"fallocate -l {size_gib}G {shlex.quote(swapfile)}")
        run(f"chmod 600 {shlex.quote(swapfile)}")
        run(f"mkswap {shlex.quote(swapfile)}")
        with open(os.path.join(target_root, "etc/fstab"), "a") as f:
            f.write(f"/swapfile\tswap\tswap\tdefaults\t0 0\n")
        return

def install_base_system_void(target_root: str):
    # Passe Pakete bei Bedarf an
    run(f"xbps-install -Suy -r {shlex.quote(target_root)} -y base-system grub efibootmgr btrfs-progs")

def install_grub(uefi: bool, target_root: str, disk_device: str, esp_mountpoint: str = "/boot/efi"):
    if uefi:
        run(f"chroot {shlex.quote(target_root)} grub-install --target=x86_64-efi --efi-directory={esp_mountpoint} --bootloader-id=Void")
    else:
        run(f"chroot {shlex.quote(target_root)} grub-install --target=i386-pc {shlex.quote(disk_device)}")
    run(f"chroot {shlex.quote(target_root)} grub-mkconfig -o /boot/grub/grub.cfg")

def enable_services_runit(target_root: str, services: List[str]):
    for svc in services:
        run(f"chroot {shlex.quote(target_root)} ln -sf /etc/sv/{shlex.quote(svc)} /var/service/{shlex.quote(svc)}", check=False)
