import gi
gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")
from gi.repository import Gtk, Adw, GLib
import shutil
import subprocess
import threading

__all__ = ["KeyboardLayoutView"]

class KeyboardLayoutView(Gtk.Box):
    """
    Schicke Tastatur-Layout-Auswahl:
      - Gtk.DropDown über Gtk.StringList (kein selected_factory nötig)
      - Live-Anwendung: zuerst 'localectl set-x11-keymap', Fallback 'setxkbmap'
      - Adw.ToastOverlay für dezente Hinweise (statt InfoBar)
      - Öffentliche API: get_selected_layout() -> 'de'/'us'/...
    """

    def __init__(self, main_app_window=None):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=0)
        self.set_margin_top(0)
        self.set_margin_bottom(0)
        self.set_margin_start(0)
        self.set_margin_end(0)

        self.main_app_window = main_app_window  # optional
        self.selected_layout_code = "de"

        # --- Overlay für Toasts (modern & dezent) ---
        overlay = Adw.ToastOverlay()
        self.append(overlay)
        self._overlay = overlay

        # --- Inhalt (gepolstert) ---
        root = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        root.set_margin_top(24)
        root.set_margin_bottom(24)
        root.set_margin_start(24)
        root.set_margin_end(24)
        self._overlay.set_child(root)

        # Titel
        title_label = Gtk.Label.new("Tastatur-Layout auswählen")
        title_label.set_halign(Gtk.Align.START)
        title_label.add_css_class("title-2")
        root.append(title_label)

        # Beschreibung
        description_label = Gtk.Label(
            label="Wählen Sie Ihr bevorzugtes Tastatur-Layout. "
                  "Die Auswahl wird sofort für die aktuelle Sitzung angewendet."
        )
        description_label.set_wrap(True)
        description_label.set_halign(Gtk.Align.START)
        root.append(description_label)

        # Auswahlzeile
        row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        row.set_halign(Gtk.Align.START)
        root.append(row)

        layout_label = Gtk.Label.new("Tastatur-Layout:")
        layout_label.set_halign(Gtk.Align.START)
        row.append(layout_label)

        # Häufige Layouts (Anzeige → XKB-Code)
        self.layouts = {
            "Deutsch (de)": "de",
            "US-Englisch (us)": "us",
            "Französisch (fr)": "fr",
            "Spanisch (es)": "es",
            "Britisch (gb)": "gb",
        }

        # DropDown Model
        self.layout_model = Gtk.StringList()
        for display_name in self.layouts.keys():
            self.layout_model.append(display_name)

        self.layout_dropdown = Gtk.DropDown(model=self.layout_model)
        self.layout_dropdown.connect("notify::selected", self._on_layout_selected)
        row.append(self.layout_dropdown)

        # Voreinstellung
        default_display = "Deutsch (de)"
        self.selected_layout_code = self.layouts.get(default_display, "de")
        self._select_display_name(default_display)

        # Testfeld
        test_label = Gtk.Label.new("Testfeld:")
        test_label.set_halign(Gtk.Align.START)
        root.append(test_label)
        self.test_entry = Gtk.Entry()
        self.test_entry.set_placeholder_text("Geben Sie hier Text ein, um das Layout zu testen…")
        root.append(self.test_entry)

        # Initial anwenden
        self._apply_layout_async(self.selected_layout_code)

    # ------------------------
    # UI / Helper
    # ------------------------
    def _select_display_name(self, name: str):
        """Selektiert einen Eintrag im DropDown nach Anzeigetext."""
        for i in range(self.layout_model.get_n_items()):
            if self.layout_model.get_string(i) == name:
                self.layout_dropdown.set_selected(i)
                return
        if self.layout_model.get_n_items() > 0:
            self.layout_dropdown.set_selected(0)

    def _toast(self, text: str):
        t = Adw.Toast.new(text)
        self._overlay.add_toast(t)

    # ------------------------
    # Events
    # ------------------------
    def _on_layout_selected(self, dropdown, _pspec):
        idx = dropdown.get_selected()
        if idx == Gtk.INVALID_LIST_POSITION:
            return
        display = self.layout_model.get_string(idx)
        code = self.layouts.get(display)
        if not code:
            self._toast("Unbekanntes Layout ausgewählt.")
            return
        self.selected_layout_code = code
        self._apply_layout_async(code)

    # ------------------------
    # Anwendung der Auswahl
    # ------------------------
    def _apply_layout_async(self, layout_code: str):
        """Wendet das Layout in einem Thread an und aktualisiert danach die UI."""
        def worker():
            ok, msg = self._apply_layout(layout_code)
            GLib.idle_add(self._after_apply, ok, msg)
        self._toast(f"Setze Layout: {layout_code} …")
        t = threading.Thread(target=worker, daemon=True)
        t.start()

    def _after_apply(self, ok: bool, msg: str):
        if ok:
            self._toast(msg)
            self.test_entry.set_text("")
            self.test_entry.grab_focus()
        else:
            self._toast(msg)

    def _apply_layout(self, layout_code: str):
        """
        Versucht erst `localectl set-x11-keymap` (funktioniert teils unter Wayland),
        fällt auf `setxkbmap` (X11) zurück.
        Gibt (ok: bool, msg: str) zurück.
        """
        # 1) localectl verfügbar?
        if shutil.which("localectl"):
            try:
                subprocess.run(
                    ["localectl", "set-x11-keymap", layout_code],
                    check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True
                )
                return True, f"Tastatur-Layout via localectl gesetzt: {layout_code}"
            except subprocess.CalledProcessError:
                pass  # Fallback versuchen

        # 2) Fallback: setxkbmap (X11)
        if shutil.which("setxkbmap"):
            try:
                subprocess.run(
                    ["setxkbmap", layout_code],
                    check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True
                )
                return True, f"Tastatur-Layout via setxkbmap gesetzt: {layout_code}"
            except subprocess.CalledProcessError as e:
                err = (e.stderr or e.stdout or "").strip()
                return False, f"Fehler beim Anwenden (setxkbmap): {err or 'Unbekannter Fehler'}"

        # Nichts verfügbar
        return False, "Weder 'localectl' noch 'setxkbmap' gefunden. Bitte X11-Tools installieren oder systemweit konfigurieren."

    # ------------------------
    # Public API
    # ------------------------
    def get_selected_layout(self) -> str:
        """Gibt den aktuell gewählten XKB-Layout-Code zurück (z. B. 'de', 'us')."""
        return self.selected_layout_code
