#!/usr/bin/env python3


import gi
gi.require_version("Gtk", "4.0")
gi.require_version("Adw", "1")
from gi.repository import Gtk, Adw, Gio, GLib, Gdk
import os

from welcome_view import WelcomeView
from partitioning_view import PartitioningView
from software_config_view import SoftwareConfigView
from summary_view import SummaryView
from installation_view import InstallationView
from completion_view import CompletionView


class InstallerApp(Adw.Application):
    def __init__(self):
        super().__init__(application_id="org.voidinstaller.App",
                         flags=Gio.ApplicationFlags.FLAGS_NONE)
        self.window: Adw.ApplicationWindow | None = None
        self.stack: Gtk.Stack | None = None
        self.view_welcome: WelcomeView | None = None
        self.view_part: PartitioningView | None = None
        self.view_software: SoftwareConfigView | None = None
        self.view_summary: SummaryView | None = None
        self.view_install: InstallationView | None = None
        self.view_done: CompletionView | None = None
        self.btn_back: Gtk.Button | None = None
        self.btn_next: Gtk.Button | None = None
        self.current_plan: dict = {}
        self.current_page: str = "welcome"

    def do_startup(self):
        Adw.Application.do_startup(self)

    def do_activate(self):
        if not self.window:
            self._load_css()
            self._build_ui()
            self._build_views()
        self.window.present()

    def _load_css(self):
        css_provider = Gtk.CssProvider()
        app_dir = os.path.dirname(os.path.abspath(__file__))
        css_file = os.path.join(app_dir, "ui.css")
        if os.path.exists(css_file):
            try:
                css_provider.load_from_path(css_file)
                display = Gdk.Display.get_default()
                Gtk.StyleContext.add_provider_for_display(
                    display, css_provider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION
                )
                print("ui.css erfolgreich geladen und angewendet.")
            except Exception as e:
                print(f"Fehler beim Laden von ui.css: {e}")
        else:
            print("Warnung: ui.css wurde nicht im Anwendungsverzeichnis gefunden.")

    def _build_ui(self):
        self.window = Adw.ApplicationWindow(application=self)
        self.window.set_title("Void Installer")
        self.window.set_default_size(1320, 1000)
        header = Adw.HeaderBar()
        self.stack = Gtk.Stack()
        self.stack.connect("notify::visible-child-name", self._on_stack_changed)
        switcher = Gtk.StackSwitcher()
        switcher.set_stack(self.stack)
        header.set_title_widget(switcher)
        nav_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        self.btn_back = Gtk.Button.new_with_label("Zurück")
        self.btn_back.connect("clicked", self.on_back)
        self.btn_next = Gtk.Button.new_with_label("Weiter")
        self.btn_next.connect("clicked", self.on_next)
        nav_box.append(self.btn_back)
        nav_box.append(self.btn_next)
        header.pack_end(nav_box)
        content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        content.append(header)
        content.append(self.stack)
        self.window.set_content(content)

    def _build_views(self):
        assert self.stack is not None
        def add_page(view, name, title):
            self.stack.add_titled(view, name, title)
            return view
        self.view_welcome = add_page(WelcomeView(), "welcome", "Willkommen")
        self.view_part = add_page(PartitioningView(), "partition", "Partitionierung")
        self.view_software = add_page(SoftwareConfigView(), "software", "Software")
        self.view_summary = add_page(SummaryView(), "summary", "Zusammenfassung")
        self.view_install = add_page(InstallationView(), "install", "Installation")
        self.view_done = add_page(CompletionView(
            on_restart_callback=self._on_restart_requested,
            on_exit_callback=self._on_exit_requested,
        ), "done", "Fertig")
        self.stack.set_visible_child_name("welcome")
        self._update_nav_for("welcome")

    def _on_stack_changed(self, *_args):
        if not self.stack: return
        name = self.stack.get_visible_child_name()
        if name: self.current_page = name; self._update_nav_for(name)

    def _update_nav_for(self, page: str):
        if not (self.btn_back and self.btn_next): return
        order = ["welcome", "partition", "software", "summary", "install", "done"]
        self.btn_back.set_sensitive(page in order[1:5])
        self.btn_back.set_visible(page not in ("welcome", "install", "done"))
        if page in ["welcome", "partition", "software"]:
            self.btn_next.set_label("Weiter"); self.btn_next.add_css_class("suggested-action"); self.btn_next.set_sensitive(True)
        elif page == "summary":
            self.btn_next.set_label("Installieren"); self.btn_next.add_css_class("suggested-action"); self.btn_next.set_sensitive(True)
        elif page == "install":
            self.btn_next.set_label("Weiter"); self.btn_next.remove_css_class("suggested-action"); self.btn_next.set_sensitive(False)
        elif page == "done":
            self.btn_next.set_visible(False)
        else:
            self.btn_next.set_label("Weiter"); self.btn_next.add_css_class("suggested-action"); self.btn_next.set_sensitive(True)

    def on_back(self, *_):
        if not self.stack: return
        order = ["welcome", "partition", "software", "summary", "install", "done"]
        idx = order.index(self.current_page) if self.current_page in order else 0
        if idx > 0 and self.current_page not in ("install", "done"):
            self.stack.set_visible_child_name(order[idx - 1])

    def on_next(self, *_):
        if not self.stack: return
        page = self.current_page
        if page == "welcome":
            try:
                if self.view_welcome: self.view_welcome.validate_user_input()
                self.stack.set_visible_child_name("partition")
            except ValueError as e: self._show_warning(str(e))
        elif page == "partition":
            try:
                if self.view_part: self.view_part.validate_plan()
                self.stack.set_visible_child_name("software")
            except ValueError as e: self._show_warning(str(e))
        elif page == "software":
            ok = self._rebuild_plan_preview_safe()
            self.stack.set_visible_child_name("summary")
            if not ok: self._show_warning("Einige Angaben konnten nicht übernommen werden.\nBitte prüfe die Felder.")
        elif page == "summary":
            self._start_installation_now()

    def _rebuild_plan_preview_safe(self) -> bool:
        try:
            plan = self._collect_plan()
            self.current_plan = plan
            if self.view_summary:

                ### KORREKTUR HIER ###
                # Wir bestimmen den Text für das Dateisystem basierend auf dem Modus
                filesystem_display = "Manuell zugewiesen"
                if plan.get("mode") in ("erase", "dual"):
                    # Wir greifen auf das Unter-Dictionary `auto_layout` zu
                    auto_layout = plan.get("auto_layout", {})
                    filesystem_display = auto_layout.get("filesystem", "Nicht festgelegt")

                self.view_summary.update_summary_data({
                    "Sprache": plan.get("language"),
                    "Zeitzone": plan.get("timezone"),
                    "Tastatur": plan.get("keyboard"),
                    "Zielgerät": plan.get("device") or "(keins gewählt)",
                    "Modus": plan.get("mode"),
                    "Dateisystem": filesystem_display, # Hier wird die korrekte Variable verwendet
                    "UEFI": "ja" if plan.get("uefi") else "nein",
                    "Benutzer": plan.get("user") or "(kein Benutzer konfiguriert)",
                    "Pakete": ", ".join(plan.get("software") or []) or "(keine)",
                })
            return True
        except Exception as e:
            if self.view_summary: self.view_summary.update_summary_data({"Hinweis": f"Plan konnte nicht aufgebaut werden: {e}"})
            return False

    def _collect_plan(self) -> dict:
        if not self.view_welcome or not self.view_part or not self.view_software: return {}
        part_plan = self.view_part.get_plan()
        user_payload = self.view_welcome.get_user_payload()
        plan = {
            **part_plan,
            "language": self.view_welcome.get_selected_language(),
            "timezone": self.view_welcome.get_selected_timezone(),
            "keyboard": self.view_welcome.get_selected_keyboard_layout(),
            "mirror": self.view_software.get_selected_mirror(),
            "software": self.view_software.get_selected_packages(),
            "include_root": self.view_software.get_include_root(),
            "user": user_payload or None,
        }
        return plan

    def _start_installation_now(self):
        plan = self.current_plan or self._collect_plan()
        if not self.view_install: return
        self.view_install.set_plan(plan); self.stack.set_visible_child_name("install"); self._update_nav_for("install")
        GLib.idle_add(lambda: (self.view_install.start_installation(), GLib.SOURCE_REMOVE)[1])

    def _on_restart_requested(self, *_):
        self.current_plan = {};
        if self.stack: self.stack.set_visible_child_name("welcome"); self._update_nav_for("welcome")

    def _on_exit_requested(self, *_): self.quit()

    def _show_warning(self, text: str):
        win = self.window or self.get_active_window()
        if isinstance(win, Adw.ApplicationWindow):
            dlg = Adw.MessageDialog.new(win, "Hinweis", text)
            dlg.add_response("ok", "OK"); dlg.set_default_response("ok"); dlg.present()

def main():
    app = InstallerApp(); app.run([])

if __name__ == "__main__":
    main()
