import gi
gi.require_version("Gtk", "4.0")
from gi.repository import Gtk, Gdk
import json
import os
import subprocess
from typing import List, Dict, Any, Optional

# --- Defaults ---
FS_CHOICES = ["btrfs", "ext4", "xfs"]
BTRFS_DEFAULT_SUBVOLS = [
    ("@", "/", True), ("@home", "/home", True),
    ("@snapshots", "/.snapshots", True), ("@var", "/var", True),
]

class PartitioningView(Gtk.Box):
    def __init__(self):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        for s in (self.set_margin_top, self.set_margin_bottom, self.set_margin_start, self.set_margin_end): s(16)
        self.is_uefi = os.path.isdir("/sys/firmware/efi")

        # --- State ---
        self.disks: List[Dict[str, Any]] = []; self.selected_disk: Optional[str] = None
        self.mode: str = "erase"; self.fs_choice: str = "btrfs"
        self.manual_assignments: Dict[str, Dict[str, Any]] = {}; self.subvol_rows = []
        self.use_separate_home: bool = False; self.home_size_percent: int = 50
        self.use_swap_partition: bool = False; self.swap_partition_gib: int = 8
        self.esp_size_mib: int = 512; self.dual_shrink_part: Optional[str] = None; self.dual_shrink_gib: int = 60

        # --- UI Aufbau ---
        title = Gtk.Label.new("Festplatten-Partitionierung"); title.add_css_class("title-2"); title.set_halign(Gtk.Align.START); self.append(title)
        sc = Gtk.ScrolledWindow(); sc.set_vexpand(True); self.append(sc)
        main_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10); sc.set_child(main_box)
        disk_frame = Gtk.Frame(label="1. Ziel-Datenträger auswählen"); disk_frame.add_css_class("card"); main_box.append(disk_frame)
        disk_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); disk_box.set_margin_top(8); disk_box.set_margin_bottom(8); disk_box.set_margin_start(12); disk_box.set_margin_end(12); disk_frame.set_child(disk_box)
        disk_box.append(Gtk.Label.new("Festplatte:")); self.device_combo = Gtk.ComboBoxText(); self.device_combo.connect("changed", self._on_device_changed); disk_box.append(self.device_combo)
        mode_frame = Gtk.Frame(label="2. Partitionierungs-Modus wählen"); mode_frame.add_css_class("card"); main_box.append(mode_frame)
        mode_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8); mode_box.set_margin_top(8); mode_box.set_margin_bottom(8); mode_box.set_margin_start(12); mode_box.set_margin_end(12); mode_frame.set_child(mode_box)
        self.rb_erase = Gtk.CheckButton.new_with_label("Automatisch: Gesamte Festplatte verwenden"); self.rb_dual  = Gtk.CheckButton.new_with_label("Automatisch: Neben bestehendem System installieren (Dualboot)"); self.rb_manual= Gtk.CheckButton.new_with_label("Manuell: Vorhandene Partitionen zuweisen")
        self.rb_dual.set_group(self.rb_erase); self.rb_manual.set_group(self.rb_erase); self.rb_erase.set_active(True)
        for rb in (self.rb_erase, self.rb_dual, self.rb_manual): rb.connect("toggled", self._on_mode_changed); mode_box.append(rb)
        self.auto_options_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10); main_box.append(self.auto_options_box)
        self.manual_options_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10); main_box.append(self.manual_options_box)
        self.dual_specific_options_box = Gtk.Frame(label="Optionen für 'Neben bestehendem System installieren'"); self.dual_specific_options_box.add_css_class("card"); self.auto_options_box.append(self.dual_specific_options_box)
        dual_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8); dual_box.set_margin_top(8); dual_box.set_margin_bottom(8); dual_box.set_margin_start(12); dual_box.set_margin_end(12); self.dual_specific_options_box.set_child(dual_box)
        row_shrink = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); dual_box.append(row_shrink); row_shrink.append(Gtk.Label.new("Partition verkleinern:")); self.dual_part_combo = Gtk.ComboBoxText(); self.dual_part_combo.connect("changed", self._on_dual_part_changed); row_shrink.append(self.dual_part_combo)
        row_size = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); dual_box.append(row_size); row_size.append(Gtk.Label.new("Neuer Platz für Void (GiB):")); self.spin_dual_gib = Gtk.SpinButton.new_with_range(10, 2048, 1); self.spin_dual_gib.set_value(self.dual_shrink_gib); self.spin_dual_gib.connect("value-changed", self._on_dual_gib_changed); row_size.append(self.spin_dual_gib)
        layout_frame = Gtk.Frame(label="3. Layout-Optionen für automatische Einrichtung"); layout_frame.add_css_class("card"); self.auto_options_box.append(layout_frame)
        layout_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10); layout_box.set_margin_top(8); layout_box.set_margin_bottom(8); layout_box.set_margin_start(12); layout_box.set_margin_end(12); layout_frame.set_child(layout_box)
        row_fs = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); row_fs.append(Gtk.Label.new("Dateisystem:")); self.fs_combo = Gtk.ComboBoxText()
        for fs in FS_CHOICES: self.fs_combo.append_text(fs)
        self.fs_combo.set_active(FS_CHOICES.index("btrfs")); self.fs_combo.connect("changed", self._on_fs_changed); row_fs.append(self.fs_combo); layout_box.append(row_fs)
        layout_box.append(Gtk.Separator(orientation=Gtk.Orientation.HORIZONTAL, margin_top=5, margin_bottom=5))
        self.home_options_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8); self.cb_home = Gtk.CheckButton.new_with_label("Eigene /home Partition anlegen"); self.cb_home.connect("toggled", self._on_home_toggled); self.home_options_box.append(self.cb_home)
        row_home_size = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); row_home_size.append(Gtk.Label.new("Größe für /home (% des freien Platzes):")); self.spin_home_percent = Gtk.SpinButton.new_with_range(5, 95, 5); self.spin_home_percent.set_value(self.home_size_percent); self.spin_home_percent.connect("value-changed", self._on_home_percent_changed); row_home_size.append(self.spin_home_percent); self.home_options_box.append(row_home_size); layout_box.append(self.home_options_box)
        self.cb_swap_part = Gtk.CheckButton.new_with_label("Swap-Partition anlegen"); self.cb_swap_part.connect("toggled", self._on_swap_part_toggled); layout_box.append(self.cb_swap_part)
        self.row_swap_size = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8); self.row_swap_size.append(Gtk.Label.new("Größe der Swap-Partition (GiB):")); self.spin_swap_part_gib = Gtk.SpinButton.new_with_range(1, 128, 1); self.spin_swap_part_gib.set_value(self.swap_partition_gib); self.spin_swap_part_gib.connect("value-changed", self._on_swap_gib_changed); self.row_swap_size.append(self.spin_swap_part_gib); layout_box.append(self.row_swap_size)
        self.subvol_frame = Gtk.Frame(label="Btrfs-Subvolumes"); self.subvol_frame.add_css_class("card"); self.auto_options_box.append(self.subvol_frame)
        subvol_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6); subvol_box.set_margin_top(8); subvol_box.set_margin_bottom(8); subvol_box.set_margin_start(12); subvol_box.set_margin_end(12); self.subvol_frame.set_child(subvol_box)
        self.subvol_list_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=4); subvol_box.append(self.subvol_list_box)
        manual_frame = Gtk.Frame(label="3. Vorhandene Partitionen verwalten"); manual_frame.add_css_class("card"); self.manual_options_box.append(manual_frame)
        manual_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8); manual_box.set_margin_top(8); manual_box.set_margin_bottom(8); manual_box.set_margin_start(12); manual_box.set_margin_end(12); manual_frame.set_child(manual_box)
        manual_box.append(Gtk.Label.new_with_mnemonic("Klicken Sie mit der _rechten Maustaste auf eine Partition, um ihr eine Rolle zuzuweisen."))
        self.partitions_listbox = Gtk.ListBox(); self.partitions_listbox.set_selection_mode(Gtk.SelectionMode.NONE); sc_list = Gtk.ScrolledWindow(); sc_list.set_child(self.partitions_listbox); sc_list.set_min_content_height(200); manual_box.append(sc_list)
        self._load_disks(); self._build_subvols_ui(); self._update_mode_visibility(); self._update_fs_options_visibility(); self._update_home_controls(); self._update_swap_controls()

    def _run_lsblk(self) -> Optional[List[Dict]]:
        try:
            out = subprocess.check_output(["lsblk", "-p", "-J", "-o", "NAME,SIZE,FSTYPE,TYPE,PATH,MOUNTPOINT"], text=True)
            return json.loads(out).get("blockdevices", [])
        except Exception as e:
            print(f"Fehler bei lsblk: {e}"); return None

    def _load_disks(self):
        self.device_combo.remove_all(); self.disks.clear()
        devices = self._run_lsblk()
        if not devices: return
        for dev_info in devices:
            if dev_info.get("type") == "disk":
                self.disks.append(dev_info); self.device_combo.append_text(f"{dev_info['path']} ({dev_info['size']})")
        if self.disks: self.device_combo.set_active(0)

    def _update_partitions_list(self, for_dualboot_combo=False):
        if for_dualboot_combo: self.dual_part_combo.remove_all()
        child = self.partitions_listbox.get_first_child()
        while child: self.partitions_listbox.remove(child); child = self.partitions_listbox.get_first_child()
        if not self.selected_disk: return
        for disk in self.disks:
            if disk.get('path') == self.selected_disk:
                for part_info in disk.get("children", []):
                    self.partitions_listbox.append(self._create_partition_row(part_info))
                    if for_dualboot_combo and part_info.get("fstype") not in [None, "swap", "vfat"]:
                        self.dual_part_combo.append_text(f"{part_info['path']} ({part_info['size']})")
                break
        if for_dualboot_combo and self.dual_part_combo.get_model() and len(self.dual_part_combo.get_model()) > 0:
            self.dual_part_combo.set_active(0)
        self._update_assignments_display()

    ### --- VOLLSTÄNDIG WIEDERHERGESTELLTE LOGIK FÜR MANUELLEN MODUS --- ###
    def _create_partition_row(self, part_info: Dict) -> Gtk.ListBoxRow:
        row = Gtk.ListBoxRow(); dev = part_info.get("path", "unbekannt")
        gesture = Gtk.GestureClick.new(); gesture.set_button(Gdk.BUTTON_SECONDARY); gesture.connect("pressed", self._on_partition_right_click, dev); row.add_controller(gesture)
        h_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12, margin_top=6, margin_bottom=6, margin_start=6, margin_end=6)
        mountpoint = f"<i>Eingehängt: {part_info['mountpoint']}</i>" if part_info.get('mountpoint') else ""
        label_text = f"<b>{dev}</b> ({part_info.get('size', '?')}) - {part_info.get('fstype', 'unbekannt')} {mountpoint}"
        label_part = Gtk.Label.new(); label_part.set_xalign(0); label_part.set_hexpand(True); label_part.set_markup(label_text); label_part.set_name("part_label"); h_box.append(label_part)
        assign_label = Gtk.Label.new(); assign_label.set_xalign(1); assign_label.set_name(f"assign_label_{dev.replace('/', '_')}"); h_box.append(assign_label)
        row.set_child(h_box)
        return row

    def _on_partition_right_click(self, gesture, n_press, x, y, device: str):
        popover = Gtk.PopoverMenu()
        menu_items = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5, margin_top=5, margin_bottom=5, margin_start=5, margin_end=5)
        mountpoints = ["/", "/home", "/boot/efi", "[SWAP]"]; group = None
        for mp in mountpoints:
            action = Gtk.CheckButton.new_with_label(f"Einhängepunkt: {mp}")
            if group: action.set_group(group)
            else: group = action
            if self.manual_assignments.get(device, {}).get("mountpoint") == mp: action.set_active(True)
            action.connect("toggled", self._on_assign_mountpoint, device, mp); menu_items.append(action)
        menu_items.append(Gtk.Separator(orientation=Gtk.Orientation.HORIZONTAL, margin_top=5, margin_bottom=5))
        format_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        format_action = Gtk.CheckButton.new_with_label("Formatieren mit:")
        fs_combo = Gtk.ComboBoxText()
        for fs in FS_CHOICES: fs_combo.append_text(fs)
        format_box.append(format_action); format_box.append(fs_combo); menu_items.append(format_box)
        format_action.connect("toggled", self._on_toggle_format, device, fs_combo)
        fs_combo.connect("changed", self._on_format_fs_changed, device)
        assignment = self.manual_assignments.get(device, {})
        if assignment.get("format"):
            format_action.set_active(True); fs_combo.set_sensitive(True)
            if assignment.get("format_fs") in FS_CHOICES: fs_combo.set_active(FS_CHOICES.index(assignment["format_fs"]))
        else: fs_combo.set_sensitive(False)
        menu_items.append(Gtk.Separator(orientation=Gtk.Orientation.HORIZONTAL, margin_top=5, margin_bottom=5))
        clear_action = Gtk.Button.new_with_label("Zuweisung löschen"); clear_action.connect("clicked", self._on_clear_assignment, device); menu_items.append(clear_action)
        popover.set_child(menu_items); popover.set_parent(gesture.get_widget()); popover.popup()

    def _update_assignments_display(self):
        child = self.partitions_listbox.get_first_child()
        while child:
            dev_path = self._get_dev_from_row(child)
            label = self._find_widget_by_name(child, f"assign_label_{dev_path.replace('/', '_')}")
            assignment = self.manual_assignments.get(dev_path)
            if label and assignment:
                mp = assignment.get("mountpoint", ""); fmt_fs = assignment.get("format_fs", "")
                fmt = f" (Formatiere als {fmt_fs})" if assignment.get("format") else ""
                label.set_markup(f"<b>{mp}{fmt}</b>")
            elif label: label.set_text("")
            child = child.get_next_sibling()

    def _get_dev_from_row(self, row):
        label = self._find_widget_by_name(row, "part_label")
        if label: return label.get_label().split(" ")[0]
        return ""

    def _find_widget_by_name(self, parent, name):
        if parent.get_name() == name: return parent
        if hasattr(parent, 'get_child') and parent.get_child():
             widget = self._find_widget_by_name(parent.get_child(), name);
             if widget: return widget
        if hasattr(parent, 'get_first_child'):
            child = parent.get_first_child()
            while child:
                widget = self._find_widget_by_name(child, name)
                if widget: return widget
                child = child.get_next_sibling()
        return None

    def _on_assign_mountpoint(self, check_button, device: str, mountpoint: str):
        if not check_button.get_active(): return
        for dev, assignment in self.manual_assignments.items():
            if assignment.get("mountpoint") == mountpoint and dev != device:
                assignment.pop("mountpoint", None)
        if device not in self.manual_assignments: self.manual_assignments[device] = {}
        self.manual_assignments[device]["mountpoint"] = mountpoint
        self._update_assignments_display()

    def _on_toggle_format(self, check_button, device: str, combo: Gtk.ComboBoxText):
        combo.set_sensitive(check_button.get_active())
        if device not in self.manual_assignments: self.manual_assignments[device] = {}
        if check_button.get_active():
            self.manual_assignments[device]["format"] = True
            if "format_fs" not in self.manual_assignments[device]: self.manual_assignments[device]["format_fs"] = combo.get_active_text() or "btrfs"
        else:
            self.manual_assignments[device].pop("format", None); self.manual_assignments[device].pop("format_fs", None)
        self._update_assignments_display()

    def _on_format_fs_changed(self, combo, device: str):
        if device in self.manual_assignments and self.manual_assignments[device].get("format"):
            self.manual_assignments[device]["format_fs"] = combo.get_active_text()
            self._update_assignments_display()

    def _on_clear_assignment(self, button, device: str):
        if device in self.manual_assignments: self.manual_assignments.pop(device)
        popover = button.get_ancestor(Gtk.Popover);
        if popover: popover.popdown()
        self._update_assignments_display()

    def _on_device_changed(self, combo):
        active_text = combo.get_active_text(); self.selected_disk = active_text.split(" ")[0] if active_text else None
        self._update_partitions_list(for_dualboot_combo=True)

    def _on_mode_changed(self, radio_button):
        if not radio_button.get_active(): return
        if self.rb_erase.get_active(): self.mode = "erase"
        elif self.rb_dual.get_active(): self.mode = "dual"
        else: self.mode = "manual"
        self._update_mode_visibility()

    def _on_fs_changed(self, combo): self.fs_choice = combo.get_active_text() or "btrfs"; self._update_fs_options_visibility()
    def _on_home_toggled(self, checkbox): self.use_separate_home = checkbox.get_active(); self._update_home_controls()
    def _on_swap_part_toggled(self, checkbox): self.use_swap_partition = checkbox.get_active(); self._update_swap_controls()
    def _on_home_percent_changed(self, spin): self.home_size_percent = int(spin.get_value())
    def _on_swap_gib_changed(self, spin): self.swap_partition_gib = int(spin.get_value())
    def _on_dual_part_changed(self, combo): active_text = combo.get_active_text(); self.dual_shrink_part = active_text.split(" ")[0] if active_text else None
    def _on_dual_gib_changed(self, spin): self.dual_shrink_gib = int(spin.get_value())

    def _update_mode_visibility(self):
        is_auto = self.mode in ("erase", "dual"); self.auto_options_box.set_visible(is_auto)
        self.manual_options_box.set_visible(self.mode == "manual")
        self.dual_specific_options_box.set_visible(self.mode == "dual")

    def _update_fs_options_visibility(self):
        is_btrfs = self.fs_choice == "btrfs"; self.subvol_frame.set_visible(is_btrfs)
        self.home_options_box.set_visible(not is_btrfs)
        if is_btrfs: self.cb_home.set_active(True)

    def _update_home_controls(self): self.spin_home_percent.set_sensitive(self.use_separate_home)
    def _update_swap_controls(self): self.row_swap_size.set_visible(self.use_swap_partition)

    def _build_subvols_ui(self):
        self.subvol_rows.clear(); child = self.subvol_list_box.get_first_child()
        while child: self.subvol_list_box.remove(child); child = self.subvol_list_box.get_first_child()
        for name, mnt, checked in BTRFS_DEFAULT_SUBVOLS:
            cb = Gtk.CheckButton.new_with_label(f"{name}  →  {mnt}"); cb.set_active(checked)
            self.subvol_list_box.append(cb); self.subvol_rows.append((name, mnt, cb))

    def get_plan(self) -> Dict[str, Any]:
        plan = {"device": self.selected_disk, "mode": self.mode, "uefi": self.is_uefi}
        if self.mode == "manual":
            plan["manual_partitions"] = self.manual_assignments
        else:
            plan["auto_layout"] = {"filesystem": self.fs_choice, "use_separate_home": self.use_separate_home if self.fs_choice != "btrfs" else False, "home_size_percent": self.home_size_percent, "use_swap_partition": self.use_swap_partition, "swap_partition_gib": self.swap_partition_gib, "esp_size_mib": self.esp_size_mib}
            if self.mode == "dual":
                plan["auto_layout"]["shrink_partition"] = self.dual_shrink_part; plan["auto_layout"]["shrink_target_gib"] = self.dual_shrink_gib
            if self.fs_choice == "btrfs":
                plan["auto_layout"]["subvolumes"] = [name for name, _, cb in self.subvol_rows if cb.get_active()]
        return plan

    def validate_plan(self):
        if not self.selected_disk: raise ValueError("Kein Ziel-Datenträger ausgewählt.")
        if self.mode == "manual" and not any(a.get("mountpoint") == "/" for a in self.manual_assignments.values()):
            raise ValueError("Manuelle Partitionierung: Sie müssen eine Partition als Wurzelverzeichnis (/) zuweisen.")
        if self.mode == "dual" and not self.dual_shrink_part:
            raise ValueError("Dualboot: Sie müssen eine zu verkleinernde Partition auswählen.")
