import gi
gi.require_version("Gtk", "4.0")
from gi.repository import Gtk
from typing import Optional, Dict

# Anzeige -> Locale-Code
LANG_MAP = {
    "Deutsch (de_DE.UTF-8)": "de_DE.UTF-8",
    "English (en_US.UTF-8)": "en_US.UTF-8",
    "Français (fr_FR.UTF-8)": "fr_FR.UTF-8",
    "Italiano (it_IT.UTF-8)": "it_IT.UTF-8",
    "Español (es_ES.UTF-8)": "es_ES.UTF-8",
}

# einfache Layout-Liste
KEYBOARD_LAYOUTS = [
    ("Deutsch (de)", "de"),
    ("Deutsch – Nodeadkeys (de-nodeadkeys)", "de-nodeadkeys"),
    ("US (us)", "us"),
    ("UK (gb)", "gb"),
    ("Französisch (fr)", "fr"),
    ("Italienisch (it)", "it"),
    ("Spanisch (es)", "es"),
]

# Eine Auswahl gängiger Zeitzonen
TIMEZONES = [
    "Europe/Berlin",
    "Europe/Zurich",
    "Europe/Vienna",
    "Europe/London",
    "Europe/Paris",
    "Europe/Rome",
    "Europe/Madrid",
    "America/New_York",
    "UTC",
]


class WelcomeView(Gtk.Box):
    """
    Startseite mit allen grundlegenden Einstellungen.
    """
    def __init__(self):
        super().__init__(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        for s in (self.set_margin_top, self.set_margin_bottom, self.set_margin_start, self.set_margin_end):
            s(16)

        # Titel
        title = Gtk.Label.new("Willkommen")
        title.add_css_class("title-1")
        title.set_halign(Gtk.Align.START)
        self.append(title)

        # Info (ohne Hinweis auf Simulation)
        info = Gtk.Label.new(
            "Wähle Sprache, Zeitzone und Tastatur-Layout. Optional kannst du schon einen Hostname und einen Benutzer anlegen."
        )
        info.set_wrap(True)
        info.set_halign(Gtk.Align.START)
        self.append(info)

        # Scrollbarer Inhalt
        sc = Gtk.ScrolledWindow()
        sc.set_vexpand(True)
        self.append(sc)

        root = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        sc.set_child(root)

        # Sprache & Zeitzone
        frm_lang = Gtk.Frame(label="Region & Sprache")
        frm_lang.add_css_class("card")
        root.append(frm_lang)

        box_lang = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        for s in (box_lang.set_margin_top, box_lang.set_margin_bottom, box_lang.set_margin_start, box_lang.set_margin_end):
            s(10)
        frm_lang.set_child(box_lang)

        # Sprachauswahl-Zeile
        row_lang = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        lang_label = Gtk.Label.new("Systemsprache:")
        lang_label.set_xalign(0)
        lang_label.set_hexpand(True)
        row_lang.append(lang_label)
        self.cmb_language = Gtk.ComboBoxText()
        for disp in LANG_MAP.keys():
            self.cmb_language.append_text(disp)
        self.cmb_language.set_active(list(LANG_MAP.keys()).index("Deutsch (de_DE.UTF-8)"))
        row_lang.append(self.cmb_language)
        box_lang.append(row_lang)

        # Zeitzonen-Auswahl
        row_tz = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        tz_label = Gtk.Label.new("Zeitzone:")
        tz_label.set_xalign(0)
        tz_label.set_hexpand(True)
        row_tz.append(tz_label)
        self.cmb_timezone = Gtk.ComboBoxText()
        for tz in TIMEZONES:
            self.cmb_timezone.append_text(tz)
        try:
            self.cmb_timezone.set_active(TIMEZONES.index("Europe/Berlin"))
        except ValueError:
            self.cmb_timezone.set_active(0)
        row_tz.append(self.cmb_timezone)
        box_lang.append(row_tz)

        # Tastatur
        frm_kbd = Gtk.Frame(label="Tastatur")
        frm_kbd.add_css_class("card")
        root.append(frm_kbd)

        box_kbd = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        for s in (box_kbd.set_margin_top, box_kbd.set_margin_bottom, box_kbd.set_margin_start, box_kbd.set_margin_end):
            s(10)
        frm_kbd.set_child(box_kbd)

        kbd_label = Gtk.Label.new("Layout:")
        kbd_label.set_xalign(0)
        kbd_label.set_hexpand(True)
        box_kbd.append(kbd_label)
        self.cmb_kbd = Gtk.ComboBoxText()
        for disp, _code in KEYBOARD_LAYOUTS:
            self.cmb_kbd.append_text(disp)
        self.cmb_kbd.set_active(0)
        box_kbd.append(self.cmb_kbd)

        # Benutzer/Host (optional)
        frm_user = Gtk.Frame(label="Benutzer & Host (optional)")
        frm_user.add_css_class("card")
        root.append(frm_user)

        box_user = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
        for s in (box_user.set_margin_top, box_user.set_margin_bottom, box_user.set_margin_start, box_user.set_margin_end):
            s(10)
        frm_user.set_child(box_user)

        # Hostname
        row_hn = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        hostname_label = Gtk.Label.new("Hostname:")
        hostname_label.set_xalign(0)
        hostname_label.set_hexpand(True)
        row_hn.append(hostname_label)
        self.entry_hostname = Gtk.Entry()
        self.entry_hostname.set_placeholder_text("z. B. voidbox")
        row_hn.append(self.entry_hostname)
        box_user.append(row_hn)

        # Username
        row_un = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        username_label = Gtk.Label.new("Benutzername:")
        username_label.set_xalign(0)
        username_label.set_hexpand(True)
        row_un.append(username_label)
        self.entry_username = Gtk.Entry()
        self.entry_username.set_placeholder_text("z. B. alice")
        row_un.append(self.entry_username)
        box_user.append(row_un)

        # Passwort
        row_pw = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        password_label = Gtk.Label.new("Passwort:")
        password_label.set_xalign(0)
        password_label.set_hexpand(True)
        row_pw.append(password_label)
        self.entry_password = Gtk.Entry()
        self.entry_password.set_visibility(False)
        self.entry_password.set_input_purpose(Gtk.InputPurpose.PASSWORD)
        self.entry_password.set_placeholder_text("Passwort")
        row_pw.append(self.entry_password)
        box_user.append(row_pw)

        # Passwort-Wiederholung
        row_pw_confirm = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
        pw_confirm_label = Gtk.Label.new("Passwort wiederholen:")
        pw_confirm_label.set_xalign(0)
        pw_confirm_label.set_hexpand(True)
        row_pw_confirm.append(pw_confirm_label)
        self.entry_password_confirm = Gtk.Entry()
        self.entry_password_confirm.set_visibility(False)
        self.entry_password_confirm.set_input_purpose(Gtk.InputPurpose.PASSWORD)
        self.entry_password_confirm.set_placeholder_text("Passwort bestätigen")
        row_pw_confirm.append(self.entry_password_confirm)
        box_user.append(row_pw_confirm)

        # Autologin
        self.chk_autologin = Gtk.CheckButton.new_with_label("Automatische Anmeldung (LightDM)")
        self.chk_autologin.set_active(False)
        box_user.append(self.chk_autologin)


    # --------- Public API für main.py ---------
    def get_selected_language(self) -> str:
        disp = self.cmb_language.get_active_text() or ""
        return LANG_MAP.get(disp, "de_DE.UTF-8")

    def get_selected_timezone(self) -> str:
        return self.cmb_timezone.get_active_text() or "Europe/Berlin"

    def get_selected_keyboard_layout(self) -> str:
        disp = self.cmb_kbd.get_active_text() or ""
        for d, code in KEYBOARD_LAYOUTS:
            if d == disp:
                return code
        return "de"

    def validate_user_input(self):
        """Prüft die Benutzereingaben, bevor die Seite gewechselt wird."""
        password = self.entry_password.get_text() or ""
        password_confirm = self.entry_password_confirm.get_text() or ""
        if password and password != password_confirm:
            raise ValueError("Die Passwörter stimmen nicht überein. Bitte überprüfen Sie Ihre Eingabe.")

    def get_user_payload(self) -> Dict:
        """Liefert optionalen Benutzer-/Host-Block."""
        out: Dict[str, object] = {}
        hostname = (self.entry_hostname.get_text() or "").strip()
        username = (self.entry_username.get_text() or "").strip()
        password = (self.entry_password.get_text() or "").strip()
        password_confirm = (self.entry_password_confirm.get_text() or "").strip()
        autologin = bool(self.chk_autologin.get_active())

        if hostname:
            out["hostname"] = hostname
        if username:
            out["name"] = username

        if password and password == password_confirm:
            out["password"] = password

        if username:
            out["groups"] = ["wheel", "audio", "video", "lp", "lpadmin", "plugdev", "bluetooth"]
        if autologin:
            out["autologin"] = True

        return out
