import gi
gi.require_version('Gtk', '4.0')
from gi.repository import Gtk, GLib, Gio, Pango
import subprocess
import threading
import sys
import os

def check_root():
    if os.geteuid() != 0:
        executable = sys.executable
        script = os.path.abspath(__file__)
        print("Root-Rechte benötigt. Starte neu mit pkexec...")
        try:
            subprocess.check_call([
                'pkexec', 'env', 
                f'DISPLAY={os.environ.get("DISPLAY", ":0")}', 
                f'XAUTHORITY={os.environ.get("XAUTHORITY", "")}', 
                executable, script
            ])
        except subprocess.CalledProcessError:
            print("Authentifizierung fehlgeschlagen.")
        sys.exit(0)

# Hilfsfunktion für Ränder (Ersatz für set_margin_all)
def set_margins(widget, margin):
    widget.set_margin_top(margin)
    widget.set_margin_bottom(margin)
    widget.set_margin_start(margin)
    widget.set_margin_end(margin)

class VoidLiveCreator(Gtk.ApplicationWindow):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.set_title("Void Linux Live ISO Creator (Extended)")
        self.set_default_size(700, 950)
        
        # Scrolled Window
        scrolled = Gtk.ScrolledWindow()
        self.set_child(scrolled)
        
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        set_margins(vbox, 20)
        scrolled.set_child(vbox)
        
        # Header
        header = Gtk.Label()
        header.set_markup("<big><b>Void Linux Live ISO Builder</b></big>")
        vbox.append(header)
        
        # --- SEKTION 1: Benutzer ---
        user_frame = Gtk.Frame(label="Benutzer-Konfiguration")
        user_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        set_margins(user_box, 10)
        user_frame.set_child(user_box)
        
        self.radio_keep = Gtk.CheckButton(label="Host-Benutzer 1:1 kopieren")
        self.radio_default = Gtk.CheckButton(label="Standard 'anon' Benutzer")
        self.radio_new = Gtk.CheckButton(label="Neuen Benutzer erstellen")
        
        self.radio_default.set_group(self.radio_keep)
        self.radio_new.set_group(self.radio_keep)
        self.radio_keep.set_active(True)
        
        user_box.append(self.radio_keep)
        user_box.append(self.radio_default)
        user_box.append(self.radio_new)
        
        # Grid für neue Benutzerdaten
        self.new_user_grid = Gtk.Grid()
        self.new_user_grid.set_column_spacing(10)
        self.new_user_grid.set_row_spacing(5)
        self.new_user_grid.set_margin_start(20)
        self.new_user_grid.set_sensitive(False)
        
        self.entry_user = Gtk.Entry(placeholder_text="Benutzername")
        self.entry_pass = Gtk.Entry(placeholder_text="Passwort")
        self.entry_pass.set_visibility(False)
        
        self.new_user_grid.attach(Gtk.Label(label="Name:"), 0, 0, 1, 1)
        self.new_user_grid.attach(self.entry_user, 1, 0, 1, 1)
        self.new_user_grid.attach(Gtk.Label(label="Passwort:"), 0, 1, 1, 1)
        self.new_user_grid.attach(self.entry_pass, 1, 1, 1, 1)
        
        user_box.append(self.new_user_grid)
        self.radio_new.connect("toggled", self.on_user_mode_changed)
        vbox.append(user_frame)

        # --- SEKTION 2: Daten & Home ---
        data_frame = Gtk.Frame(label="Daten & Verzeichnisse")
        data_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        set_margins(data_box, 10)
        data_frame.set_child(data_box)
        
        self.chk_full_home = Gtk.CheckButton(label="Komplettes /home verzeichnis kopieren")
        self.chk_full_home.connect("toggled", self.on_home_mode_changed)
        data_box.append(self.chk_full_home)
        
        # Granulare Auswahl
        lbl_select = Gtk.Label(label="<b>Oder einzelne Ordner wählen:</b>", xalign=0)
        lbl_select.set_use_markup(True)
        data_box.append(lbl_select)
        
        self.granular_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=2)
        self.granular_box.set_margin_start(20)
        
        self.chk_docs = Gtk.CheckButton(label="Dokumente")
        self.chk_pics = Gtk.CheckButton(label="Bilder")
        self.chk_down = Gtk.CheckButton(label="Downloads")
        self.chk_conf = Gtk.CheckButton(label=".config (Einstellungen)")
        self.chk_local = Gtk.CheckButton(label=".local (Daten/Binaries)")
        
        for chk in [self.chk_docs, self.chk_pics, self.chk_down, self.chk_conf, self.chk_local]:
            self.granular_box.append(chk)
            
        data_box.append(self.granular_box)
        
        # --- ÄNDERUNG HIER: Button zum Auswählen hinzugefügt ---
        data_box.append(Gtk.Separator())
        data_box.append(Gtk.Label(label="Zusätzliche Pfade (vom Host):", xalign=0))
        
        path_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        
        self.entry_custom_path = Gtk.Entry(placeholder_text="/opt/my-app")
        self.entry_custom_path.set_hexpand(True)
        
        # Button zum Durchsuchen
        btn_browse_path = Gtk.Button(label="Ordner...")
        btn_browse_path.connect("clicked", self.on_browse_custom_path)
        
        # Button zum Hinzufügen
        btn_add_path = Gtk.Button(label="Hinzufügen")
        btn_add_path.connect("clicked", self.on_add_path)
        
        path_box.append(self.entry_custom_path)
        path_box.append(btn_browse_path)
        path_box.append(btn_add_path)
        data_box.append(path_box)
        
        # Liste (TreeView)
        self.path_store = Gtk.ListStore(str)
        self.path_view = Gtk.TreeView(model=self.path_store)
        renderer = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn("Gewählte Pfade", renderer, text=0)
        self.path_view.append_column(col)
        self.path_view.set_size_request(-1, 80)
        
        path_scroll = Gtk.ScrolledWindow()
        path_scroll.set_child(self.path_view)
        path_scroll.set_min_content_height(80)
        data_box.append(path_scroll)
        
        vbox.append(data_frame)
        
        # --- SEKTION 3: Overlay / Include ---
        overlay_frame = Gtk.Frame(label="System Overlay (Include Ordner)")
        overlay_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        set_margins(overlay_box, 10)
        overlay_frame.set_child(overlay_box)
        
        lbl_overlay = Gtk.Label(label="Ordnerstruktur hier wählen, die über das Root-FS gelegt wird:", xalign=0)
        overlay_box.append(lbl_overlay)
        
        ov_chooser_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        self.entry_overlay = Gtk.Entry(placeholder_text="Kein Overlay gewählt")
        self.entry_overlay.set_hexpand(True)
        btn_ov = Gtk.Button(label="Wählen...")
        btn_ov.connect("clicked", self.on_choose_overlay)
        ov_chooser_box.append(self.entry_overlay)
        ov_chooser_box.append(btn_ov)
        overlay_box.append(ov_chooser_box)
        
        vbox.append(overlay_frame)

        # --- Output & Action ---
        action_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
        
        # Output File
        out_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        out_box.append(Gtk.Label(label="ISO Ausgabe:"))
        self.entry_out = Gtk.Entry(text="/var/tmp/void-custom.iso")
        self.entry_out.set_hexpand(True)
        out_box.append(self.entry_out)
        action_box.append(out_box)
        
        # Progress
        self.progress = Gtk.ProgressBar(show_text=True)
        action_box.append(self.progress)
        self.lbl_status = Gtk.Label(label="Bereit")
        action_box.append(self.lbl_status)
        
        # Buttons
        btn_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5, halign=Gtk.Align.END)
        self.btn_create = Gtk.Button(label="ISO Erstellen")
        self.btn_create.add_css_class("suggested-action")
        self.btn_create.connect("clicked", self.on_create)
        btn_box.append(self.btn_create)
        action_box.append(btn_box)
        
        vbox.append(action_box)

    def on_user_mode_changed(self, btn):
        self.new_user_grid.set_sensitive(self.radio_new.get_active())
        
    def on_home_mode_changed(self, btn):
        active = btn.get_active()
        self.granular_box.set_sensitive(not active)

    # --- Neue Funktion: Pfad Dialog öffnen ---
    def on_browse_custom_path(self, btn):
        dialog = Gtk.FileDialog()
        dialog.select_folder(self, None, self.on_custom_path_selected)

    # --- Neue Funktion: Pfad Dialog Ergebnis ---
    def on_custom_path_selected(self, dialog, result):
        try:
            folder = dialog.select_folder_finish(result)
            if folder:
                self.entry_custom_path.set_text(folder.get_path())
        except:
            pass

    def on_add_path(self, btn):
        txt = self.entry_custom_path.get_text()
        if txt and os.path.exists(txt):
            # Prüfen ob schon in der Liste
            exists = False
            for row in self.path_store:
                if row[0] == txt:
                    exists = True
                    break
            
            if not exists:
                self.path_store.append([txt])
                self.entry_custom_path.set_text("") # Leeren nach Hinzufügen
            else:
                self.lbl_status.set_text("Pfad ist bereits in der Liste.")
        else:
            self.lbl_status.set_text("Pfad existiert nicht oder ist leer!")

    def on_choose_overlay(self, btn):
        dialog = Gtk.FileDialog()
        dialog.select_folder(self, None, self.on_overlay_selected)

    def on_overlay_selected(self, dialog, result):
        try:
            f = dialog.select_folder_finish(result)
            if f: self.entry_overlay.set_text(f.get_path())
        except: pass

    def on_create(self, btn):
        self.btn_create.set_sensitive(False)
        self.progress.set_fraction(0.0)
        
        # Optionen sammeln
        custom_paths = [row[0] for row in self.path_store]
        
        opts = {
            'output': self.entry_out.get_text(),
            'user_mode': 'new' if self.radio_new.get_active() else ('keep' if self.radio_keep.get_active() else 'default'),
            'new_user': self.entry_user.get_text(),
            'new_pass': self.entry_pass.get_text(),
            'full_home': self.chk_full_home.get_active(),
            'home_parts': {
                'Documents': self.chk_docs.get_active(),
                'Pictures': self.chk_pics.get_active(),
                'Downloads': self.chk_down.get_active(),
                '.config': self.chk_conf.get_active(),
                '.local': self.chk_local.get_active()
            },
            'custom_paths': custom_paths,
            'overlay_path': self.entry_overlay.get_text() if self.entry_overlay.get_text() else None
        }
        
        thread = threading.Thread(target=self.run_builder, args=(opts,))
        thread.daemon = True
        thread.start()

    def run_builder(self, opts):
        sys.path.append(os.path.dirname(os.path.abspath(__file__)))
        try:
            from iso_builder import ISOBuilder
            builder = ISOBuilder(opts)
            for p, m in builder.build():
                GLib.idle_add(self.update_ui, p, m)
            GLib.idle_add(self.done, True)
        except Exception as e:
            GLib.idle_add(self.update_ui, 0, f"Error: {e}")
            GLib.idle_add(self.done, False)

    def update_ui(self, p, m):
        self.progress.set_fraction(p)
        self.lbl_status.set_text(m)

    def done(self, success):
        self.btn_create.set_sensitive(True)
        if success:
            dialog = Gtk.MessageDialog(transient_for=self, modal=True, message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK, text="Fertig!")
            dialog.format_secondary_text(f"ISO erstellt unter:\n{self.entry_out.get_text()}")
            dialog.connect("response", lambda d,r: d.destroy())
            dialog.present()

class App(Gtk.Application):
    def __init__(self): super().__init__(application_id='org.void.iso')
    def do_activate(self): VoidLiveCreator(application=self).present()

if __name__ == '__main__':
    check_root()
    App().run(None)